/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvPatchgpuField

Description
    Abstract base class with a fat-interface to all derived classes
    covering all possible ways in which they might be used.

    The first level of derivation is to basic patchFields which cover
    zero-gradient, fixed-gradient, fixed-value and mixed conditions.

    The next level of derivation covers all the specialised types with
    specific evaluation procedures, particularly with respect to specific
    fields.

SourceFiles
    fvPatchgpuField.C
    fvPatchgpuFieldNew.C

\*---------------------------------------------------------------------------*/

#ifndef fvPatchgpuField_H
#define fvPatchgpuField_H

#include "gpufvPatch.H"
#include "DimensionedgpuField.H"
#include "fieldTypes.H"
#include "scalarField.H"
#include "fvPatchField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward Declarations

class objectRegistry;
class dictionary;
class fvPatchgpuFieldMapper;
class gpuvolMesh;

template<class Type> class fvPatchgpuField;
template<class Type> class calculatedFvPatchgpuField;
template<class Type> class gpufvMatrix;
template<class Type> class fvPatchField;


template<class Type>
Ostream& operator<<(Ostream&, const fvPatchgpuField<Type>&);


/*---------------------------------------------------------------------------*\
                        Class fvPatchgpuField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class fvPatchgpuField
:
    public gpuField<Type>
{
    // Private Data

        //- Reference to patch
        const gpufvPatch& patch_;

        //- Reference to internal field
        const DimensionedgpuField<Type, gpuvolMesh>& internalField_;

        //- Update index used so that updateCoeffs is called only once during
        //- the construction of the matrix
        bool updated_;

        //- Update index used so that manipulateMatrix is called only once
        //- during the construction of the matrix
        bool manipulatedMatrix_;

        //- Use implicit formulation
        bool useImplicit_;

        //- Optional patch type, used to allow specified boundary conditions
        //  to be applied to constraint patches by providing the constraint
        //  patch type as 'patchType'
        word patchType_;


public:

    typedef gpufvPatch Patch;
    typedef calculatedFvPatchgpuField<Type> Calculated;
//	typedef fvPatchField hostType;


    //- Runtime type information
    TypeName("fvPatchgpuField");

    //- Debug switch to disallow the use of genericFvPatchField
    static int disallowGenericFvPatchgpuField;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            fvPatchgpuField,
            patch,
            (
                const gpufvPatch& p,
                const DimensionedgpuField<Type, gpuvolMesh>& iF
            ),
            (p, iF)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            fvPatchgpuField,
            patchMapper,
            (
                const fvPatchgpuField<Type>& ptf,
                const gpufvPatch& p,
                const DimensionedgpuField<Type, gpuvolMesh>& iF,
                const fvPatchgpuFieldMapper& m
            ),
            (dynamic_cast<const fvPatchgpuFieldType&>(ptf), p, iF, m)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            fvPatchgpuField,
            dictionary,
            (
                const gpufvPatch& p,
                const DimensionedgpuField<Type, gpuvolMesh>& iF,
                const dictionary& dict
            ),
            (p, iF, dict)
        );


    // Constructors

        //- Construct from patch and internal field
        fvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and value
        fvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const Type& value
        );

        //- Construct from patch and internal field and patch type
        fvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const word& patchType
        );

        //- Construct from patch and internal field and patch field
        fvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const Field<Type>&
        );

        //- Construct from patch and internal field and patch field
        fvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const gpuField<Type>&
        );

        //- Construct from patch, internal field and dictionary
        fvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&,
            const bool valueRequired=true
        );

        //- Construct by mapping the given fvPatchgpuField onto a new patch
        fvPatchgpuField
        (
            const fvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        fvPatchgpuField(const fvPatchgpuField<Type>&);

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>::New(*this);
        }

        //- Construct as copy setting internal field reference
        fvPatchgpuField
        (
            const fvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>::New(*this, iF);
        }


    // Selectors

        //- Return a pointer to a new patchField created on freestore given
        //  patch and internal field
        //  (does not set the patch field values)
        static tmp<fvPatchgpuField<Type>> New
        (
            const word&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Return a pointer to a new patchField created on freestore given
        //  patch and internal field
        //  (does not set the patch field values).
        //  Allows override of constraint type
        static tmp<fvPatchgpuField<Type>> New
        (
            const word&,
            const word& actualPatchType,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Return a pointer to a new patchField created on freestore from
        //  a given fvPatchgpuField mapped onto a new patch
        static tmp<fvPatchgpuField<Type>> New
        (
            const fvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Return a pointer to a new patchField created on freestore
        //  from dictionary
        static tmp<fvPatchgpuField<Type>> New
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Return a pointer to a new calculatedFvPatchField created on
        //  freestore without setting patchField values
        static tmp<fvPatchgpuField<Type>> NewCalculatedType
        (
            const gpufvPatch&
        );

        //- Return a pointer to a new calculatedFvPatchField created on
        //  freestore without setting patchField values
        template<class Type2>
        static tmp<fvPatchgpuField<Type>> NewCalculatedType
        (
            const fvPatchgpuField<Type2>&
        );


    //- Destructor
    virtual ~fvPatchgpuField<Type>() = default;


    // Member Functions

        // Implicit Functions

            //- Use implicit formulation for coupled patches only
            bool useImplicit() const noexcept
            {
                return useImplicit_;
            }

            //- Set useImplicit on/off
            //  \return old value
            bool useImplicit(bool on) noexcept
            {
                bool old(useImplicit_);
                useImplicit_ = on;
                return old;
            }


        // Attributes

            //- Return the type of the calculated for of fvPatchgpuField
            static const word& calculatedType();

            //- Return true if this patch field fixes a value.
            //  Needed to check if a level has to be specified while solving
            //  Poissons equations.
            virtual bool fixesValue() const
            {
                return false;
            }

            //- Return true if the value of the patch field
            //  is altered by assignment (the default)
            virtual bool assignable() const
            {
                return true;
            }

            //- Return true if this patch field is coupled
            virtual bool coupled() const
            {
                return false;
            }


        // Access

            //- Return local objectRegistry
            const objectRegistry& db() const;

            //- Return patch
            const gpufvPatch& patch() const
            {
                return patch_;
            }

            //- Return dimensioned internal field reference
            const DimensionedgpuField<Type, gpuvolMesh>& internalField() const
            {
                return internalField_;
            }

            //- Return internal field reference
            const gpuField<Type>& primitiveField() const
            {
                return internalField_;
            }

            //- Optional patch type
            const word& patchType() const
            {
                return patchType_;
            }

            //- Optional patch type
            word& patchType()
            {
                return patchType_;
            }

            //- Return true if the boundary condition has already been updated
            bool updated() const
            {
                return updated_;
            }

            //- Return true if the matrix has already been manipulated
            bool manipulatedMatrix() const
            {
                return manipulatedMatrix_;
            }


        // Mapping Functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap
            (
                const fvPatchgpuFieldMapper&
            );

            //- Reverse map the given fvPatchgpuField onto this fvPatchgpuField
            virtual void rmap
            (
                const fvPatchgpuField<Type>&,
                const labelgpuList&
            );


        // Evaluation Functions

            //- Return patch-normal gradient
            virtual tmp<gpuField<Type>> snGrad() const;

            //- Return patch-normal gradient for coupled-patches
            //  using the deltaCoeffs provided
            virtual tmp<gpuField<Type>> snGrad
            (
                const scalargpuField& deltaCoeffs
            ) const
            {
                NotImplemented;
                return *this;
            }

            //- Update the coefficients associated with the patch field
            //  Sets Updated to true
            virtual void updateCoeffs();

            //- Update the coefficients associated with the patch field
            //  with a weight field (0..1). This weight field is usually
            //  provided as the amount of geometric overlap for 'duplicate'
            //  patches. Sets Updated to true
            virtual void updateWeightedCoeffs(const scalargpuField& weights);

            //- Return internal field next to patch as patch field
            virtual tmp<gpuField<Type>> patchInternalField() const;

            //- Return internal field next to patch as patch field
            virtual void patchInternalField(gpuField<Type>&) const;

            //- Return patchField on the opposite patch of a coupled patch
            virtual tmp<gpuField<Type>> patchNeighbourField() const
            {
                NotImplemented;
                return *this;
            }

            //- Initialise the evaluation of the patch field
            virtual void initEvaluate
            (
                const Pstream::commsTypes commsType =
                    Pstream::commsTypes::blocking
            )
            {}

            //- Evaluate the patch field, sets Updated to false
            virtual void evaluate
            (
                const Pstream::commsTypes commsType =
                    Pstream::commsTypes::blocking
            );


            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the value of this patchField with given weights
            virtual tmp<gpuField<Type>> valueInternalCoeffs
            (
                const tmp<gpuField<scalar>>&
            ) const
            {
                NotImplemented;
                return *this;
            }

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the value of this patchField with given weights
            virtual tmp<gpuField<Type>> valueBoundaryCoeffs
            (
                const tmp<gpuField<scalar>>&
            ) const
            {
                NotImplemented;
                return *this;
            }

            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the gradient of this patchField
            virtual tmp<gpuField<Type>> gradientInternalCoeffs() const
            {
                NotImplemented;
                return *this;
            }

            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the gradient of this coupled patchField
            //  using the deltaCoeffs provided
            virtual tmp<gpuField<Type>> gradientInternalCoeffs
            (
                const scalargpuField& deltaCoeffs
            ) const
            {
                NotImplemented;
                return *this;
            }

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the gradient of this patchField
            virtual tmp<gpuField<Type>> gradientBoundaryCoeffs() const
            {
                NotImplemented;
                return *this;
            }

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the gradient of this coupled patchField
            //  using the deltaCoeffs provided
            virtual tmp<gpuField<Type>> gradientBoundaryCoeffs
            (
                const scalargpuField& deltaCoeffs
            ) const
            {
                NotImplemented;
                return *this;
            }


            //- Manipulate matrix
            virtual void manipulateMatrix(gpufvMatrix<Type>& matrix);

            //- Manipulate matrix with given weights
            virtual void manipulateMatrix
            (
                gpufvMatrix<Type>& matrix,
                const scalargpuField& weights
            );

            //- Manipulate fvMatrix
            virtual void manipulateMatrix
            (
                gpufvMatrix<Type>& matrix,
                const label iMatrix,
                const direction cmp
            );


        // I-O

            //- Write
            virtual void write(Ostream&) const;


        // Check

            //- Check fvPatchgpuField<Type> against given fvPatchgpuField<Type>
            void check(const fvPatchgpuField<Type>&) const;


    // Member Operators

        virtual void operator=(const UList<Type>&);
        virtual void operator=(const gpuList<Type>&);

        virtual void operator=(const fvPatchgpuField<Type>&);
        virtual void operator+=(const fvPatchgpuField<Type>&);
        virtual void operator-=(const fvPatchgpuField<Type>&);
        virtual void operator*=(const fvPatchgpuField<scalar>&);
        virtual void operator/=(const fvPatchgpuField<scalar>&);

        virtual void operator+=(const gpuField<Type>&);
        virtual void operator-=(const gpuField<Type>&);

        virtual void operator*=(const gpuField<scalar>&);
        virtual void operator/=(const gpuField<scalar>&);

        virtual void operator=(const Type&);
        virtual void operator+=(const Type&);
        virtual void operator-=(const Type&);
        virtual void operator*=(const scalar);
        virtual void operator/=(const scalar);


        // Force an assignment irrespective of form of patch

        virtual void operator==(const fvPatchgpuField<Type>&);
        virtual void operator==(const gpuField<Type>&);
        virtual void operator==(const Type&);


    // Ostream operator

        friend Ostream& operator<< <Type>(Ostream&, const fvPatchgpuField<Type>&);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "fvPatchgpuField.C"
    #include "calculatedFvPatchgpuField.H"
#endif


#define makeFvPatchField(fvPatchTypeField)                                     \
                                                                               \
defineNamedTemplateTypeNameAndDebug(fvPatchTypeField, 0);                      \
template<>                                                                     \
int fvPatchTypeField::disallowGenericFvPatchgpuField                              \
(                                                                              \
    debug::debugSwitch("disallowGenericFvPatchgpuField", 0)                       \
);                                                                             \
defineTemplateRunTimeSelectionTable(fvPatchTypeField, patch);                  \
defineTemplateRunTimeSelectionTable(fvPatchTypeField, patchMapper);            \
defineTemplateRunTimeSelectionTable(fvPatchTypeField, dictionary);


#define addToPatchFieldRunTimeSelection(PatchTypeField, typePatchTypeField)    \
    addToRunTimeSelectionTable                                                 \
    (                                                                          \
        PatchTypeField,                                                        \
        typePatchTypeField,                                                    \
        patch                                                                  \
    );                                                                         \
    addToRunTimeSelectionTable                                                 \
    (                                                                          \
        PatchTypeField,                                                        \
        typePatchTypeField,                                                    \
        patchMapper                                                            \
    );                                                                         \
    addToRunTimeSelectionTable                                                 \
    (                                                                          \
        PatchTypeField,                                                        \
        typePatchTypeField,                                                    \
        dictionary                                                             \
    );


// Use with caution
#define addRemovableToPatchFieldRunTimeSelection\
(PatchTypeField, typePatchTypeField)                                           \
                                                                               \
    addRemovableToRunTimeSelectionTable                                        \
    (                                                                          \
        PatchTypeField,                                                        \
        typePatchTypeField,                                                    \
        patch                                                                  \
    );                                                                         \
    addRemovableToRunTimeSelectionTable                                        \
    (                                                                          \
        PatchTypeField,                                                        \
        typePatchTypeField,                                                    \
        patchMapper                                                            \
    );                                                                         \
    addRemovableToRunTimeSelectionTable                                        \
    (                                                                          \
        PatchTypeField,                                                        \
        typePatchTypeField,                                                    \
        dictionary                                                             \
    );


// For non-templated patch fields
#define makePatchTypeField(PatchTypeField, typePatchTypeField)                 \
    defineTypeNameAndDebug(typePatchTypeField, 0);                             \
    addToPatchFieldRunTimeSelection(PatchTypeField, typePatchTypeField)

// For non-templated patch fields - use with caution
#define makeRemovablePatchTypeField(PatchTypeField, typePatchTypeField)        \
    defineTypeNameAndDebug(typePatchTypeField, 0);                             \
    addRemovableToPatchFieldRunTimeSelection(PatchTypeField, typePatchTypeField)

// For templated patch fields
#define makeTemplatePatchTypeField(fieldType, type)                            \
    defineNamedTemplateTypeNameAndDebug                                        \
    (                                                                          \
        CAT4(type, FvPatch, CAPITALIZE(fieldType), gpuField),                     \
        0                                                                      \
    );                                                                         \
    addToPatchFieldRunTimeSelection                                            \
    (                                                                          \
        CAT3(fvPatch, CAPITALIZE(fieldType), gpuField),                           \
        CAT4(type, FvPatch, CAPITALIZE(fieldType), gpuField)                      \
    )

#define makePatchFields(type)                                                  \
    FOR_ALL_FIELD_TYPES(makeTemplatePatchTypeField, type)

#define makePatchFieldTypeName(fieldType, type)                                \
    defineNamedTemplateTypeNameAndDebug                                        \
    (                                                                          \
        CAT4(type, FvPatch, CAPITALIZE(fieldType), gpuField),                     \
        0                                                                      \
    );

#define makePatchFieldTypeNames(type)                                          \
    FOR_ALL_FIELD_TYPES(makePatchFieldTypeName, type)

#define makePatchTypeFieldTypedef(fieldType, type)                             \
    typedef type##FvPatchgpuField<fieldType>                                      \
        CAT4(type, FvPatch, CAPITALIZE(fieldType), gpuField);

#define makePatchTypeFieldTypedefs(type)                                       \
    FOR_ALL_FIELD_TYPES(makePatchTypeFieldTypedef, type)

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
