/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "turbulentIntensityKineticEnergyInletFvPatchScalargpuField.H"
#include "addToRunTimeSelectionTable.H"
#include "fvPatchgpuFieldMapper.H"
#include "surfacegpuFields.H"
#include "volgpuFields.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::
turbulentIntensityKineticEnergyInletFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    inletOutletFvPatchScalargpuField(p, iF),
    intensity_(0.0),
    UName_("U")
{
    this->refValue() = 0.0;
    this->refGrad() = 0.0;
    this->valueFraction() = 0.0;
}

Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::
turbulentIntensityKineticEnergyInletFvPatchScalargpuField
(
    const turbulentIntensityKineticEnergyInletFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    inletOutletFvPatchScalargpuField(ptf, p, iF, mapper),
    intensity_(ptf.intensity_),
    UName_(ptf.UName_)
{}

Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::
turbulentIntensityKineticEnergyInletFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    inletOutletFvPatchScalargpuField(p, iF),
    intensity_(dict.get<scalar>("intensity")),
    UName_(dict.getOrDefault<word>("U", "U"))
{
    this->patchType() = dict.getOrDefault<word>("patchType", word::null);
    this->phiName_ = dict.getOrDefault<word>("phi", "phi");

    if (intensity_ < 0 || intensity_ > 1)
    {
        FatalErrorInFunction
            << "Turbulence intensity should be specified as a fraction 0-1 "
               "of the mean velocity\n"
               "    value given is " << intensity_ << nl
            << "    on patch " << this->patch().name()
            << " of field " << this->internalField().name()
            << " in file " << this->internalField().objectPath()
            << exit(FatalError);
    }

    fvPatchScalargpuField::operator=(scalargpuField("value", dict, p.size()));

    this->refValue() = 0.0;
    this->refGrad() = 0.0;
    this->valueFraction() = 0.0;
}

Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::
turbulentIntensityKineticEnergyInletFvPatchScalargpuField
(
    const turbulentIntensityKineticEnergyInletFvPatchScalargpuField& ptf
)
:
    inletOutletFvPatchScalargpuField(ptf),
    intensity_(ptf.intensity_),
    UName_(ptf.UName_)
{}


Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::
turbulentIntensityKineticEnergyInletFvPatchScalargpuField
(
    const turbulentIntensityKineticEnergyInletFvPatchScalargpuField& ptf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    inletOutletFvPatchScalargpuField(ptf, iF),
    intensity_(ptf.intensity_),
    UName_(ptf.UName_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::
updateCoeffs()
{
    if (updated())
    {
        return;
    }

    const fvPatchVectorgpuField& Up =
        patch().lookupPatchField<volVectorgpuField, vector>(UName_);

    const fvsPatchScalargpuField& phip =
        patch().lookupPatchField<surfaceScalargpuField, scalar>(this->phiName_);

    this->refValue() = 1.5*sqr(intensity_)*magSqr(Up);
    this->valueFraction() = 1.0 - pos0(phip);

    inletOutletFvPatchScalargpuField::updateCoeffs();
}


void Foam::turbulentIntensityKineticEnergyInletFvPatchScalargpuField::write
(
    Ostream& os
) const
{
    fvPatchScalargpuField::write(os);
    os.writeEntry("intensity", intensity_);
    os.writeEntryIfDifferent<word>("U", "U", UName_);
    os.writeEntryIfDifferent<word>("phi", "phi", this->phiName_);
    writeEntry("value", os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        turbulentIntensityKineticEnergyInletFvPatchScalargpuField
    );
}

// ************************************************************************* //
