/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "totalPressureFvPatchScalargpuField.H"
#include "addToRunTimeSelectionTable.H"
#include "fvPatchgpuFieldMapper.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::totalPressureFvPatchScalargpuField::totalPressureFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedValueFvPatchScalargpuField(p, iF),
    UName_("U"),
    phiName_("phi"),
    rhoName_("rho"),
    psiName_("none"),
    gamma_(0.0),
    p0_(p.size(), Zero)
{}


Foam::totalPressureFvPatchScalargpuField::totalPressureFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    fixedValueFvPatchScalargpuField(p, iF, dict, false),
    UName_(dict.getOrDefault<word>("U", "U")),
    phiName_(dict.getOrDefault<word>("phi", "phi")),
    rhoName_(dict.getOrDefault<word>("rho", "rho")),
    psiName_(dict.getOrDefault<word>("psi", "none")),
    gamma_(psiName_ != "none" ? dict.get<scalar>("gamma") : 1),
    p0_("p0", dict, p.size())
{
    if (dict.found("value"))
    {
        fvPatchgpuField<scalar>::operator=
        (
            scalargpuField("value", dict, p.size())
        );
    }
    else
    {
        fvPatchgpuField<scalar>::operator=(p0_);
    }
}


Foam::totalPressureFvPatchScalargpuField::totalPressureFvPatchScalargpuField
(
    const totalPressureFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    fixedValueFvPatchScalargpuField(ptf, p, iF, mapper),
    UName_(ptf.UName_),
    phiName_(ptf.phiName_),
    rhoName_(ptf.rhoName_),
    psiName_(ptf.psiName_),
    gamma_(ptf.gamma_),
    p0_(ptf.p0_, mapper)
{}


Foam::totalPressureFvPatchScalargpuField::totalPressureFvPatchScalargpuField
(
    const totalPressureFvPatchScalargpuField& tppsf
)
:
    fixedValueFvPatchScalargpuField(tppsf),
    UName_(tppsf.UName_),
    phiName_(tppsf.phiName_),
    rhoName_(tppsf.rhoName_),
    psiName_(tppsf.psiName_),
    gamma_(tppsf.gamma_),
    p0_(tppsf.p0_)
{}


Foam::totalPressureFvPatchScalargpuField::totalPressureFvPatchScalargpuField
(
    const totalPressureFvPatchScalargpuField& tppsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedValueFvPatchScalargpuField(tppsf, iF),
    UName_(tppsf.UName_),
    phiName_(tppsf.phiName_),
    rhoName_(tppsf.rhoName_),
    psiName_(tppsf.psiName_),
    gamma_(tppsf.gamma_),
    p0_(tppsf.p0_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::totalPressureFvPatchScalargpuField::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    fixedValueFvPatchScalargpuField::autoMap(m);
    p0_.autoMap(m);
}


void Foam::totalPressureFvPatchScalargpuField::rmap
(
    const fvPatchScalargpuField& ptf,
    const labelgpuList& addr
)
{
    fixedValueFvPatchScalargpuField::rmap(ptf, addr);

    const totalPressureFvPatchScalargpuField& tiptf =
        refCast<const totalPressureFvPatchScalargpuField>(ptf);

    p0_.rmap(tiptf.p0_, addr);
}


void Foam::totalPressureFvPatchScalargpuField::updateCoeffs
(
    const scalargpuField& p0p,
    const vectorgpuField& Up
)
{
    if (updated())
    {
        return;
    }

    const fvsPatchgpuField<scalar>& phip =
        patch().lookupPatchField<surfaceScalargpuField, scalar>(phiName_);

    if (internalField().dimensions() == dimPressure)
    {
        if (psiName_ == "none")
        {
            // Variable density and low-speed compressible flow

            const fvPatchgpuField<scalar>& rho =
                patch().lookupPatchField<volScalargpuField, scalar>(rhoName_);

            operator==(p0p - 0.5*rho*(1.0 - pos0(phip))*magSqr(Up));
        }
        else
        {
            // High-speed compressible flow

            const fvPatchgpuField<scalar>& psip =
                patch().lookupPatchField<volScalargpuField, scalar>(psiName_);

            if (gamma_ > 1)
            {
                scalar gM1ByG = (gamma_ - 1)/gamma_;

                operator==
                (
                    p0p
                   /pow
                    (
                        (1.0 + 0.5*psip*gM1ByG*(1.0 - pos0(phip))*magSqr(Up)),
                        1.0/gM1ByG
                    )
                );
            }
            else
            {
                operator==(p0p/(1.0 + 0.5*psip*(1.0 - pos0(phip))*magSqr(Up)));
            }
        }

    }
    else if (internalField().dimensions() == dimPressure/dimDensity)
    {
        // Incompressible flow
        operator==(p0p - 0.5*(1.0 - pos0(phip))*magSqr(Up));
    }
    else
    {
        FatalErrorInFunction
            << " Incorrect pressure dimensions " << internalField().dimensions()
            << nl
            << "    Should be " << dimPressure
            << " for compressible/variable density flow" << nl
            << "    or " << dimPressure/dimDensity
            << " for incompressible flow," << nl
            << "    on patch " << this->patch().name()
            << " of field " << this->internalField().name()
            << " in file " << this->internalField().objectPath()
            << exit(FatalError);
    }

    fixedValueFvPatchScalargpuField::updateCoeffs();
}


void Foam::totalPressureFvPatchScalargpuField::updateCoeffs()
{
    updateCoeffs
    (
        p0(),
        patch().lookupPatchField<volVectorgpuField, vector>(UName())
    );
}


void Foam::totalPressureFvPatchScalargpuField::write(Ostream& os) const
{
    fvPatchScalargpuField::write(os);
    os.writeEntryIfDifferent<word>("U", "U", UName_);
    os.writeEntryIfDifferent<word>("phi", "phi", phiName_);
    os.writeEntry("rho", rhoName_);
    os.writeEntry("psi", psiName_);
    os.writeEntry("gamma", gamma_);
    p0_.writeEntry("p0", os);
    writeEntry("value", os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        totalPressureFvPatchScalargpuField
    );
}

// ************************************************************************* //
