/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::partialSlipFvPatchField

Group
    grpWallBoundaryConditions grpGenericBoundaryConditions

Description
    This boundary condition provides a partial slip condition.  The amount of
    slip is controlled by a user-supplied field.

Usage
    \table
      Property      | Description                     | Reqd | Default
      type          | Type name: partialSlip          | yes  | -
      refValue      | Reference value at zero slip    | no   | 0
      valueFraction | Fraction of refValue used for boundary [0-1] | yes |
      writeValue    | Output patch value (eg, ParaView) | no | false
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            partialSlip;
        refValue        uniform 0.001;
        valueFraction   uniform 0.1;
        value           uniform 0;
    }
    \endverbatim

See also
    Foam::transformFvPatchField

SourceFiles
    partialSlipFvPatchField.C

\*---------------------------------------------------------------------------*/

#ifndef partialSlipFvPatchgpuField_H
#define partialSlipFvPatchgpuField_H

#include "transformFvPatchgpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                  Class partialSlipFvPatchField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class partialSlipFvPatchgpuField
:
    public transformFvPatchgpuField<Type>
{
    //- The parent boundary condition type
    typedef transformFvPatchgpuField<Type> parent_bctype;


    // Private Data

        //- Reference value
        gpuField<Type> refValue_;

        //- Fraction (0-1) of value used for boundary condition
        scalargpuField valueFraction_;

        //- Flag to output patch values (e.g. for ParaView)
        bool writeValue_;


public:

    //- Runtime type information
    TypeName("partialSlip");


    // Constructors

        //- Construct from patch and internal field
        partialSlipFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        partialSlipFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given partialSlipFvPatchField onto a new patch
        partialSlipFvPatchgpuField
        (
            const partialSlipFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        partialSlipFvPatchgpuField
        (
            const partialSlipFvPatchgpuField<Type>&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new partialSlipFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        partialSlipFvPatchgpuField
        (
            const partialSlipFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new partialSlipFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member Functions

        // Access/attributes

            //- This patch field is not altered by assignment
            virtual bool assignable() const
            {
                return false;
            }

            virtual gpuField<Type>& refValue()
            {
                return refValue_;
            }

            virtual const gpuField<Type>& refValue() const
            {
                return refValue_;
            }

            virtual scalargpuField& valueFraction()
            {
                return valueFraction_;
            }

            virtual const scalargpuField& valueFraction() const
            {
                return valueFraction_;
            }


        // Mapping

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap
            (
                const fvPatchgpuFieldMapper&
            );

            //- Reverse map the given fvPatchField onto this fvPatchField
            virtual void rmap
            (
                const fvPatchgpuField<Type>&,
                const labelgpuList&
            );


        // Evaluation

            //- Return gradient at boundary
            virtual tmp<gpuField<Type>> snGrad() const;

            //- Evaluate the patch field
            virtual void evaluate
            (
                const Pstream::commsTypes commsType =
                    Pstream::commsTypes::blocking
            );

            //- Return face-gradient transform diagonal
            virtual tmp<gpuField<Type>> snGradTransformDiag() const;


        //- Write
        virtual void write(Ostream&) const;


    // Member Operators

        virtual void operator=(const UList<Type>&) {}
        virtual void operator=(const gpuList<Type>&) {}

        virtual void operator=(const fvPatchgpuField<Type>&) {}
        virtual void operator+=(const fvPatchgpuField<Type>&) {}
        virtual void operator-=(const fvPatchgpuField<Type>&) {}
        virtual void operator*=(const fvPatchgpuField<scalar>&) {}
        virtual void operator/=(const fvPatchgpuField<scalar>&) {}

        virtual void operator+=(const gpuField<Type>&) {}
        virtual void operator-=(const gpuField<Type>&) {}

        virtual void operator*=(const gpuField<scalar>&) {}
        virtual void operator/=(const gpuField<scalar>&) {}

        virtual void operator=(const Type&) {}
        virtual void operator+=(const Type&) {}
        virtual void operator-=(const Type&) {}
        virtual void operator*=(const scalar) {}
        virtual void operator/=(const scalar) {}
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "partialSlipFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
