/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2017-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "partialSlipFvPatchgpuField.H"
#include "symmTransformField.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Type>
Foam::partialSlipFvPatchgpuField<Type>::partialSlipFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    parent_bctype(p, iF),
    refValue_(p.size(), Zero),
    valueFraction_(p.size(), 1.0),
    writeValue_(false)
{}


template<class Type>
Foam::partialSlipFvPatchgpuField<Type>::partialSlipFvPatchgpuField
(
    const partialSlipFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    parent_bctype(ptf, p, iF, mapper),
    refValue_(ptf.refValue_, mapper),
    valueFraction_(ptf.valueFraction_, mapper),
    writeValue_(ptf.writeValue_)
{}


template<class Type>
Foam::partialSlipFvPatchgpuField<Type>::partialSlipFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    parent_bctype(p, iF),
    refValue_(p.size(), Zero),
    valueFraction_("valueFraction", dict, p.size()),
    writeValue_(dict.getOrDefault("writeValue", false))
{
    this->patchType() = dict.getOrDefault<word>("patchType", word::null);

    // Backwards compatibility - leave refValue as zero unless specified
    if (dict.found("refValue"))
    {
        refValue_ = gpuField<Type>("refValue", dict, p.size());
    }

    evaluate();
}


template<class Type>
Foam::partialSlipFvPatchgpuField<Type>::partialSlipFvPatchgpuField
(
    const partialSlipFvPatchgpuField<Type>& ptf
)
:
    parent_bctype(ptf),
    refValue_(ptf.refValue_),
    valueFraction_(ptf.valueFraction_),
    writeValue_(ptf.writeValue_)
{}


template<class Type>
Foam::partialSlipFvPatchgpuField<Type>::partialSlipFvPatchgpuField
(
    const partialSlipFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    parent_bctype(ptf, iF),
    refValue_(ptf.refValue_),
    valueFraction_(ptf.valueFraction_),
    writeValue_(ptf.writeValue_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
void Foam::partialSlipFvPatchgpuField<Type>::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    parent_bctype::autoMap(m);
    refValue_.autoMap(m);
    valueFraction_.autoMap(m);
}


template<class Type>
void Foam::partialSlipFvPatchgpuField<Type>::rmap
(
    const fvPatchgpuField<Type>& ptf,
    const labelgpuList& addr
)
{
    parent_bctype::rmap(ptf, addr);

    const auto& dmptf =
        refCast<const partialSlipFvPatchgpuField<Type>>(ptf);

    refValue_.rmap(dmptf.refValue_, addr);
    valueFraction_.rmap(dmptf.valueFraction_, addr);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::partialSlipFvPatchgpuField<Type>::snGrad() const
{
    tmp<vectorgpuField> nHat = this->patch().nf();
    const gpuField<Type> pif(this->patchInternalField());

    return
    (
        valueFraction_*refValue_
      + (1.0 - valueFraction_)*transform(I - sqr(nHat), pif) - pif
    )*this->patch().deltaCoeffs();
}


template<class Type>
void Foam::partialSlipFvPatchgpuField<Type>::evaluate
(
    const Pstream::commsTypes
)
{
    if (!this->updated())
    {
        this->updateCoeffs();
    }

    tmp<vectorgpuField> nHat = this->patch().nf();

    gpuField<Type>::operator=
    (
        valueFraction_*refValue_
      +
        (1.0 - valueFraction_)
       *transform(I - sqr(nHat), this->patchInternalField())
    );

    parent_bctype::evaluate();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::partialSlipFvPatchgpuField<Type>::snGradTransformDiag() const
{
    const vectorgpuField nHat(this->patch().nf());
    vectorgpuField diag(nHat.size());

    diag.replace(vector::X, mag(nHat.component(vector::X)));
    diag.replace(vector::Y, mag(nHat.component(vector::Y)));
    diag.replace(vector::Z, mag(nHat.component(vector::Z)));

    return
        valueFraction_*pTraits<Type>::one
      + (1.0 - valueFraction_)
       *transformFieldMask<Type>(pow<vector, pTraits<Type>::rank>(diag));
}


template<class Type>
void Foam::partialSlipFvPatchgpuField<Type>::write(Ostream& os) const
{
    this->parent_bctype::write(os);
    refValue_.writeEntry("refValue", os);
    valueFraction_.writeEntry("valueFraction", os);

    if (writeValue_)
    {
        os.writeEntry("writeValue", "true");
        this->writeEntry("value", os);
    }
}


// ************************************************************************* //
