/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::outletPhaseMeanVelocityFvPatchVectorField

Group
    grpOutletBoundaryConditions

Description
    This boundary condition adjusts the velocity for the given phase to achieve
    the specified mean thus causing the phase-fraction to adjust according to
    the mass flow rate.

    Typical usage is as the outlet condition for a towing-tank ship simulation
    to maintain the outlet water level at the level as the inlet.

Usage
    \table
        Property     | Description             | Required    | Default value
        Umean        | mean velocity normal to the boundary [m/s] | yes |
        alpha        | phase-fraction field    | yes |
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            outletPhaseMeanVelocity;
        Umean           1.2;
        alpha           alpha.water;
        value           uniform (1.2 0 0);
    }
    \endverbatim

See also
    Foam::mixedFvPatchField
    Foam::variableHeightFlowRateInletVelocityFvPatchVectorField

SourceFiles
    outletPhaseMeanVelocityFvPatchVectorField.C

\*---------------------------------------------------------------------------*/

#ifndef outletPhaseMeanVelocityFvPatchVectorgpuField_H
#define outletPhaseMeanVelocityFvPatchVectorgpuField_H

#include "mixedFvPatchgpuFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
/*---------------------------------------------------------------------------*\
    Class outletPhaseMeanVelocityFvPatchVectorField Declaration
\*---------------------------------------------------------------------------*/

class outletPhaseMeanVelocityFvPatchVectorgpuField
:
    public mixedFvPatchVectorgpuField
{
    // Private data

        //- Inlet integral flow rate
        scalar Umean_;

        //- Name of the phase-fraction field
        word alphaName_;


public:

   //- Runtime type information
   TypeName("outletPhaseMeanVelocity");


   // Constructors

        //- Construct from patch and internal field
        outletPhaseMeanVelocityFvPatchVectorgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<vector, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        outletPhaseMeanVelocityFvPatchVectorgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<vector, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given
        //  outletPhaseMeanVelocityFvPatchVectorField
        //  onto a new patch
        outletPhaseMeanVelocityFvPatchVectorgpuField
        (
            const outletPhaseMeanVelocityFvPatchVectorgpuField&,
            const gpufvPatch&,
            const DimensionedgpuField<vector, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        outletPhaseMeanVelocityFvPatchVectorgpuField
        (
            const outletPhaseMeanVelocityFvPatchVectorgpuField&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchVectorgpuField> clone() const
        {
            return tmp<fvPatchVectorgpuField>
            (
                new outletPhaseMeanVelocityFvPatchVectorgpuField(*this)
            );
        }

        //- Construct as copy setting internal field reference
        outletPhaseMeanVelocityFvPatchVectorgpuField
        (
            const outletPhaseMeanVelocityFvPatchVectorgpuField&,
            const DimensionedgpuField<vector, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchVectorgpuField> clone
        (
            const DimensionedgpuField<vector, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchVectorgpuField>
            (
                new outletPhaseMeanVelocityFvPatchVectorgpuField
                (
                    *this,
                    iF
                )
            );
        }


    // Member functions

        // Access

            //- Return the flux
            scalar Umean() const
            {
                return Umean_;
            }

            //- Return reference to the flux to allow adjustment
            scalar& Umean()
            {
                return Umean_;
            }


        //- Update the coefficients associated with the patch field
        virtual void updateCoeffs();

        //- Write
        virtual void write(Ostream&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
