/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fixedJumpAMIFvPatchField

Group
    grpCoupledBoundaryConditions

Description
    This boundary condition provides a jump condition, across non-conformal
    cyclic path-pairs, employing an arbitraryMeshInterface (AMI).

    The jump is specified as a fixed value field, applied as an offset to the
    'owner' patch.

Usage
    \table
        Property     | Description             | Required    | Default value
        patchType    | underlying patch type should be \c cyclic| yes |
        jump         | current jump value      | yes         |
    \endtable

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            fixedJumpAMI;
        patchType       cyclic;
        jump            uniform 10;
    }
    \endverbatim

    The above example shows the use of a fixed jump of '10'.

Note
     The underlying \c patchType should be set to \c cyclicAMI

See also
    Foam::jumpCyclicAMIFvPatchField

SourceFiles
    fixedJumpAMIFvPatchField.C

\*---------------------------------------------------------------------------*/

#ifndef fixedJumpAMIFvPatchgpuField_H
#define fixedJumpAMIFvPatchgpuField_H

#include "jumpCyclicAMIFvPatchgpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                  Class fixedJumpAMIFvPatchField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class fixedJumpAMIFvPatchgpuField
:
    public jumpCyclicAMIFvPatchgpuField<Type>
{

protected:

    // Protected data

        //- "jump" field
        gpuField<Type> jump_;


public:

    //- Runtime type information
    TypeName("fixedJumpAMI");


    // Constructors

        //- Construct from patch and internal field
        fixedJumpAMIFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        fixedJumpAMIFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given fixedJumpAMIFvPatchField onto a
        //  new patch
        fixedJumpAMIFvPatchgpuField
        (
            const fixedJumpAMIFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        fixedJumpAMIFvPatchgpuField
        (
            const fixedJumpAMIFvPatchgpuField<Type>&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new fixedJumpAMIFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        fixedJumpAMIFvPatchgpuField
        (
            const fixedJumpAMIFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new fixedJumpAMIFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Access

            //- Return the "jump" across the patch
            virtual tmp<gpuField<Type>> jump() const;


        // Mapping functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap(const fvPatchgpuFieldMapper&);

            //- Reverse map the given fvPatchField onto this fvPatchField
            virtual void rmap(const fvPatchgpuField<Type>&, const labelgpuList&);


        //- Write
        virtual void write(Ostream&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "fixedJumpAMIFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
