/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2016-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fixedFluxPressureFvPatchScalarField

Group
    grpInletBoundaryConditions grpWallBoundaryConditions

Description
    This boundary condition sets the pressure gradient to the provided value
    such that the flux on the boundary is that specified by the velocity
    boundary condition.

    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        type            fixedFluxPressure;
    }
    \endverbatim

See also
    Foam::fixedGradientFvPatchField

SourceFiles
    fixedFluxPressureFvPatchScalarField.C

\*---------------------------------------------------------------------------*/

#ifndef fixedFluxPressureFvPatchScalargpuField_H
#define fixedFluxPressureFvPatchScalargpuField_H

#include "fvPatchgpuFields.H"
#include "fixedGradientFvPatchgpuFields.H"
#include "gpuupdateableSnGrad.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
             Class fixedFluxPressureFvPatchScalarField Declaration
\*---------------------------------------------------------------------------*/

class fixedFluxPressureFvPatchScalargpuField
:
    public fixedGradientFvPatchScalargpuField,
    public gpuupdateablePatchTypes::updateableSnGrad//
{
    // Private data

        //- Current time index (used for updating)
        label curTimeIndex_;


public:

    //- Runtime type information
    TypeName("fixedFluxPressure");


    // Constructors

        //- Construct from patch and internal field
        fixedFluxPressureFvPatchScalargpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        fixedFluxPressureFvPatchScalargpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given fixedFluxPressureFvPatchScalarField onto
        //  a new patch
        fixedFluxPressureFvPatchScalargpuField
        (
            const fixedFluxPressureFvPatchScalargpuField&,
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        fixedFluxPressureFvPatchScalargpuField
        (
            const fixedFluxPressureFvPatchScalargpuField&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchScalargpuField> clone() const
        {
            return tmp<fvPatchScalargpuField>
            (
                new fixedFluxPressureFvPatchScalargpuField(*this)
            );
        }

        //- Construct as copy setting internal field reference
        fixedFluxPressureFvPatchScalargpuField
        (
            const fixedFluxPressureFvPatchScalargpuField&,
            const DimensionedgpuField<scalar, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchScalargpuField> clone
        (
            const DimensionedgpuField<scalar, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchScalargpuField>
            (
                new fixedFluxPressureFvPatchScalargpuField(*this, iF)
            );
        }


    // Member functions

        //- Update the patch pressure gradient field from the given snGradp
        virtual void updateSnGrad(const scalargpuField& snGradp);

        //- Update the patch pressure gradient field
        virtual void updateCoeffs();

        //- Write
        virtual void write(Ostream&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "volgpuFields.H"

namespace Foam
{
    template<class GradBC>
    inline void setSnGrad
    (
        volScalargpuField::Boundary& bf,
        const FieldField<fvsPatchgpuField, scalar>& snGrad
    )
    {
        forAll(bf, patchi)
        {
            if (isA<GradBC>(bf[patchi]))
            {
                refCast<GradBC>(bf[patchi]).updateSnGrad(snGrad[patchi]);
            }
        }
    }

    template<class GradBC>
    inline void setSnGrad
    (
        volScalargpuField::Boundary& bf,
        const tmp<FieldField<fvsPatchgpuField, scalar>>& tsnGrad
    )
    {
        setSnGrad<GradBC>(bf, tsnGrad());
    }
}

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
