/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "fixedGradientFvPatchgpuField.H"
#include "dictionary.H"

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
Foam::fixedGradientFvPatchgpuField<Type>::fixedGradientFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    fvPatchgpuField<Type>(p, iF),
    gradient_(p.size(), Zero)
{}


template<class Type>
Foam::fixedGradientFvPatchgpuField<Type>::fixedGradientFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    fvPatchgpuField<Type>(p, iF, dict, false),
    gradient_("gradient", dict, p.size())
{
    evaluate();
}


template<class Type>
Foam::fixedGradientFvPatchgpuField<Type>::fixedGradientFvPatchgpuField
(
    const fixedGradientFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    fvPatchgpuField<Type>(ptf, p, iF, mapper),
    gradient_(ptf.gradient_, mapper)
{
    if (notNull(iF) && mapper.hasUnmapped())
    {
        WarningInFunction
            << "On field " << iF.name() << " patch " << p.name()
            << " patchField " << this->type()
            << " : mapper does not map all values." << nl
            << "    To avoid this warning fully specify the mapping in derived"
            << " patch fields." << endl;
    }
}


template<class Type>
Foam::fixedGradientFvPatchgpuField<Type>::fixedGradientFvPatchgpuField
(
    const fixedGradientFvPatchgpuField<Type>& ptf
)
:
    fvPatchgpuField<Type>(ptf),
    gradient_(ptf.gradient_)
{}


template<class Type>
Foam::fixedGradientFvPatchgpuField<Type>::fixedGradientFvPatchgpuField
(
    const fixedGradientFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    fvPatchgpuField<Type>(ptf, iF),
    gradient_(ptf.gradient_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
void Foam::fixedGradientFvPatchgpuField<Type>::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    fvPatchgpuField<Type>::autoMap(m);
    gradient_.autoMap(m);
}


template<class Type>
void Foam::fixedGradientFvPatchgpuField<Type>::rmap
(
    const fvPatchgpuField<Type>& ptf,
    const labelgpuList& addr
)
{
    fvPatchgpuField<Type>::rmap(ptf, addr);

    const fixedGradientFvPatchgpuField<Type>& fgptf =
        refCast<const fixedGradientFvPatchgpuField<Type>>(ptf);

    gradient_.rmap(fgptf.gradient_, addr);
}


template<class Type>
void Foam::fixedGradientFvPatchgpuField<Type>::evaluate(const Pstream::commsTypes)
{
    if (!this->updated())
    {
        this->updateCoeffs();
    }

    gpuField<Type>::operator=
    (
        this->patchInternalField() + gradient_/this->patch().deltaCoeffs()
    );

    fvPatchgpuField<Type>::evaluate();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::fixedGradientFvPatchgpuField<Type>::valueInternalCoeffs
(
    const tmp<scalargpuField>&
) const
{
    return tmp<gpuField<Type>>::New(this->size(), pTraits<Type>::one);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::fixedGradientFvPatchgpuField<Type>::valueBoundaryCoeffs
(
    const tmp<scalargpuField>&
) const
{
    return gradient()/this->patch().deltaCoeffs();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::fixedGradientFvPatchgpuField<Type>::gradientInternalCoeffs() const
{
    return tmp<gpuField<Type>>::New(this->size(), Zero);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::fixedGradientFvPatchgpuField<Type>::gradientBoundaryCoeffs() const
{
    return gradient();
}


template<class Type>
void Foam::fixedGradientFvPatchgpuField<Type>::write(Ostream& os) const
{
    fvPatchgpuField<Type>::write(os);
    gradient_.writeEntry("gradient", os);
}


// ************************************************************************* //
