/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::snGradScheme

Group
    grpFvSnGradSchemes

Description
    Abstract base class for runtime selected \c snGrad surface
    normal gradient schemes.

    A surface normal gradient is evaluated at a cell face. It
    is the normal-to-face component of the gradient of
    values at the centres of two cells that the face connects.

    Unit-surface-normal vector decomposition is based on the
    so-called over-relaxed approach. Orthogonal components are
    treated implicitly and non-orthogonal components are treated
    explicitly with (or without) various limiters.

SourceFiles
    snGradScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpusnGradScheme_H
#define gpusnGradScheme_H

#include "tmp.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                        Class snGradScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpusnGradScheme
:
    public refCount
{
    // Private Data

        //- Hold const reference to mesh
        const gpufvMesh& mesh_;


    // Private Member Functions

        //- No copy construct
        gpusnGradScheme(const gpusnGradScheme&) = delete;

        //- No copy assignment
        void operator=(const gpusnGradScheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpusnGradScheme,
            Mesh,
            (const gpufvMesh& mesh, Istream& schemeData),
            (mesh, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpusnGradScheme(const gpufvMesh& mesh)
        :
            mesh_(mesh)
        {}


    // Selectors

        //- Return new tmp interpolation scheme
        static tmp<gpusnGradScheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpusnGradScheme() = default;


    // Member Functions

        //- Return const reference to mesh
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        //- Return the snGrad of the given cell field
        //- by using the given deltaCoeffs
        static tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        snGrad
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const tmp<surfaceScalargpuField>&,
            const word& snGradName = "snGrad"
        );

        //- Return the sndGrad of the given cell field
        static tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        sndGrad
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const word& snGradName = "sndGrad"
        );

        //- Return the interpolation weighting factors for the given field
        virtual tmp<surfaceScalargpuField> deltaCoeffs
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const noexcept
        {
            return false;
        }

        //- Return the explicit correction to the snGrad
        //- for the given field
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&) const
        {
            return tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
            (
                nullptr
            );
        }

        //- Return the snGrad of the given cell field
        //- with explicit correction
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        snGrad(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&) const;

        //- Return the snGrad of the given tmp cell field
        //- with explicit correction
        tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        snGrad
        (
            const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>&
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makegpuSnGradTypeScheme(SS, Type)                                         \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpusnGradScheme<Type>::addMeshConstructorToTable<SS<Type>>            \
                add##SS##Type##MeshConstructorToTable_;                        \
        }                                                                      \
    }

#define makegpuSnGradScheme(SS)                                                   \
                                                                               \
makegpuSnGradTypeScheme(SS, scalar)                                               \
makegpuSnGradTypeScheme(SS, vector)                                               \
makegpuSnGradTypeScheme(SS, sphericalTensor)                                      \
makegpuSnGradTypeScheme(SS, symmTensor)                                           \
makegpuSnGradTypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpusnGradScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
