/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "GPCG.H"
#include "PrecisionAdaptor.H"

#include "lduMatrixSolverFunctors.H"
#include "PCGCache.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(GPCG, 0);

    gpulduMatrix::solver::addsymMatrixConstructorToTable<GPCG>
        addGPCGSymMatrixConstructorToTable_;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::GPCG::GPCG
(
    const word& fieldName,
    const gpulduMatrix& matrix,
    const FieldField<gpuField, scalar>& interfaceBouCoeffs,
    const FieldField<gpuField, scalar>& interfaceIntCoeffs,
    const lduInterfacegpuFieldPtrsList& interfaces,
    const dictionary& solverControls
)
:
    gpulduMatrix::solver
    (
        fieldName,
        matrix,
        interfaceBouCoeffs,
        interfaceIntCoeffs,
        interfaces,
        solverControls
    )
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::solverPerformance Foam::GPCG::scalarSolve
(
    scalargpuField& psi,
    const scalargpuField& source,
    const direction cmpt
) const
{    
    // --- Setup class containing solver performance data
    solverPerformance solverPerf
    (
        gpulduMatrix::preconditioner::getName(controlDict_) + typeName,
        fieldName_
    );

    label nCells = psi.size();
    scalargpuField wA(PCGCache::wA(matrix_.level(),nCells),nCells);
    scalargpuField pA(PCGCache::pA(matrix_.level(),nCells),nCells);

    matrix_.Amul(wA, psi, interfaceBouCoeffs_, interfaces_, cmpt);

    solveScalar wArA = solverPerf.great_;
    solveScalar wArAold = wArA;

    scalargpuField rA(PCGCache::rA(matrix_.level(),nCells),nCells);
    thrust::transform
    (
        source.begin(),
        source.end(),
        wA.begin(),
        rA.begin(),
        minusOp<scalar>()
    );

    solveScalar normFactor = this->normFactor(psi, source, wA, pA);

    //solveScalarField rA(nCells);
    //thrust::copy(grA.begin(),grA.end(),rA.begin());
    //matrix().setResidualField
    //(
    //    ConstPrecisionAdaptor<scalar, solveScalar>(rA)(),
    //    fieldName_,
    //    true
    //);

    if ((log_ >= 2) || (gpulduMatrix::debug >= 2))
    {
        Info<< "   Normalisation factor = " << normFactor << endl;
    }

    // --- Calculate normalised residual norm
    solverPerf.initialResidual() =
        gSumMag(rA, matrix().mesh().comm())
       /normFactor;
    solverPerf.finalResidual() = solverPerf.initialResidual();

    // --- Check convergence, solve if not converged
    if
    (
        minIter_ > 0
     || !solverPerf.checkConvergence(tolerance_, relTol_, log_)
    )
    {
        // --- Select and construct the preconditioner
        autoPtr<gpulduMatrix::preconditioner> preconPtr =
            gpulduMatrix::preconditioner::New
            (
                *this,
                controlDict_
            );

        // --- Solver iteration
        do
        {
            // --- Store previous wArA
            wArAold = wArA;

            // --- Precondition residual
            preconPtr->precondition(wA, rA, cmpt);

            // --- Update search directions:
            wArA = gSumProd(wA, rA, matrix().mesh().comm());

            if (solverPerf.nIterations() == 0)
            {
                thrust::copy(wA.begin(),wA.end(),pA.begin());
            }
            else
            {
                solveScalar beta = wArA/wArAold;
				
                thrust::transform
                (
                    wA.begin(),
                    wA.end(),
                    pA.begin(),
                    pA.begin(),
                    wAPlusBetaPAFunctor(beta)
                );
            }

            // --- Update preconditioned residual
            matrix_.Amul(wA, pA, interfaceBouCoeffs_, interfaces_, cmpt);

            solveScalar wApA = gSumProd(wA, pA, matrix().mesh().comm());

            // --- Test for singularity
            if (solverPerf.checkSingularity(mag(wApA)/normFactor)) break;

            // --- Update solution and residual:

            solveScalar alpha = wArA/wApA;
		
            thrust::transform
            (
                psi.begin(),
                psi.end(),
                pA.begin(),
                psi.begin(),
                psiPlusAlphaPAFunctor(alpha)
            );

            thrust::transform
            (
                rA.begin(),
                rA.end(),
                wA.begin(),
                rA.begin(),
                rAMinusAlphaWAFunctor(alpha)
            );

            solverPerf.finalResidual() =
                gSumMag(rA, matrix().mesh().comm())
               /normFactor;

        } while
        (
            (
              ++solverPerf.nIterations() < maxIter_
            && !solverPerf.checkConvergence(tolerance_, relTol_, log_)
            )
         || solverPerf.nIterations() < minIter_
        );
    }

    //rA(grA);
    //thrust::copy(grA.begin(),grA.end(),rA.begin());
    //matrix().setResidualField
    //(
    //    ConstPrecisionAdaptor<scalar, solveScalar>(rA)(),
    //    fieldName_,
    //    false
    //);

    return solverPerf;
}

Foam::solverPerformance Foam::GPCG::solve
(
    scalargpuField& psi_s,
    const scalargpuField& source,
    const direction cmpt
) const
{
    return scalarSolve
    (
        psi_s,
        source,
        cmpt
    );
}

// ************************************************************************* //
