/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "processorGGAMGInterface.H"
#include "addToRunTimeSelectionTable.H"
#include "labelPairHashes.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(processorGGAMGInterface, 0);
    addToRunTimeSelectionTable
    (
        GGAMGInterface,
        processorGGAMGInterface,
        gpulduInterface
    );
    addToRunTimeSelectionTable
    (
        GGAMGInterface,
        processorGGAMGInterface,
        Istream
    );
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::processorGGAMGInterface::processorGGAMGInterface
(
    const label index,
    const gpulduInterfacePtrsList& coarseInterfaces,
    const gpulduInterface& fineInterface,
    const labelField& localRestrictAddressing,
    const labelField& neighbourRestrictAddressing,
    const label fineLevelIndex,
    const label coarseComm
)
:
    GGAMGInterface
    (
        index,
        coarseInterfaces
    ),
    comm_(coarseComm),
    myProcNo_(refCast<const processorgpuLduInterface>(fineInterface).myProcNo()),
    neighbProcNo_
    (
        refCast<const processorgpuLduInterface>(fineInterface).neighbProcNo()
    ),
    forwardT_(refCast<const processorgpuLduInterface>(fineInterface).gpuForwardT()),
    gForwardT_(forwardT_),
    tag_(refCast<const processorgpuLduInterface>(fineInterface).tag())
{
    // From coarse face to coarse cell
    DynamicList<label> dynFaceCells(localRestrictAddressing.size());
    // From fine face to coarse face
    DynamicList<label> dynFaceRestrictAddressing
    (
        localRestrictAddressing.size()
    );

    // From coarse cell pair to coarse face
    labelPairLookup cellsToCoarseFace(2*localRestrictAddressing.size());

    forAll(localRestrictAddressing, ffi)
    {
        labelPair cellPair;

        // Do switching on master/slave indexes based on the owner/neighbour of
        // the processor index such that both sides get the same answer.
        if (myProcNo() < neighbProcNo())
        {
            // Master side
            cellPair = labelPair
            (
                localRestrictAddressing[ffi],
                neighbourRestrictAddressing[ffi]
            );
        }
        else
        {
            // Slave side
            cellPair = labelPair
            (
                neighbourRestrictAddressing[ffi],
                localRestrictAddressing[ffi]
            );
        }

        const auto fnd = cellsToCoarseFace.cfind(cellPair);

        if (fnd.found())
        {
            // Already have coarse face
            dynFaceRestrictAddressing.append(fnd.val());
        }
        else
        {
            // New coarse face
            label coarseI = dynFaceCells.size();
            dynFaceRestrictAddressing.append(coarseI);
            dynFaceCells.append(localRestrictAddressing[ffi]);
            cellsToCoarseFace.insert(cellPair, coarseI);
        }
    }

    faceCellsHost_.transfer(dynFaceCells);
    faceRestrictAddressingHost_.transfer(dynFaceRestrictAddressing);
    faceCells_ = faceCellsHost_;

    updateAddressing();
}


Foam::processorGGAMGInterface::processorGGAMGInterface
(
    const label index,
    const gpulduInterfacePtrsList& coarseInterfaces,
    const labelUList& faceCells,
    const labelUList& faceRestrictAddresssing,
    const label coarseComm,
    const label myProcNo,
    const label neighbProcNo,
    const tensorField& forwardT,
    const int tag
)
:
    GGAMGInterface
    (
        index,
        coarseInterfaces,
        faceCells,
        faceRestrictAddresssing
    ),
    comm_(coarseComm),
    myProcNo_(myProcNo),
    neighbProcNo_(neighbProcNo),
    forwardT_(forwardT),
    gForwardT_(forwardT_),
    tag_(tag)
{}


Foam::processorGGAMGInterface::processorGGAMGInterface
(
    const label index,
    const gpulduInterfacePtrsList& coarseInterfaces,
    Istream& is
)
:
    GGAMGInterface(index, coarseInterfaces, is),
    comm_(readLabel(is)),
    myProcNo_(readLabel(is)),
    neighbProcNo_(readLabel(is)),
    forwardT_(is),
    gForwardT_(forwardT_),
    tag_(readLabel(is))
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::processorGGAMGInterface::initInternalFieldTransfer
(
    const Pstream::commsTypes commsType,
    const labelgpuList& iF
) const
{
    send(commsType, interfaceInternalField(iF)());
}


void Foam::processorGGAMGInterface::initInternalFieldTransfer
(
    const Pstream::commsTypes commsType,
    const labelgpuList& iF,
    const labelgpuList& faceCells
) const
{
    send(commsType, interfaceInternalField(iF, faceCells)());
}


Foam::tmp<Foam::labelgpuField> Foam::processorGGAMGInterface::internalFieldTransfer
(
    const Pstream::commsTypes commsType,
    const labelgpuList& iF
) const
{
    tmp<labelgpuField> tfld(receive<label>(commsType, this->size()));

    return tfld;
}


void Foam::processorGGAMGInterface::write(Ostream& os) const
{
    GGAMGInterface::write(os);
    os  << token::SPACE << comm_
        << token::SPACE << myProcNo_
        << token::SPACE << neighbProcNo_
        << token::SPACE << forwardT_
        << token::SPACE << tag_;
}


// ************************************************************************* //
