/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::GAMGInterface

Description
    Abstract base class for GAMG agglomerated interfaces.

SourceFiles
    GAMGInterface.C
    newAmgInterface.C

\*---------------------------------------------------------------------------*/

#ifndef GGAMGInterface_H
#define GGAMGInterface_H

#include "autoPtr.H"
#include "gpulduInterfacePtrsList.H"
#include "GGAMGAgglomeration.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class GAMGInterface Declaration
\*---------------------------------------------------------------------------*/

class GGAMGInterface
:
    public gpulduInterface
{
protected:

    // Protected data

        //- My index in coarseInterfaces
        const label index_;

        //- All interfaces
        const gpulduInterfacePtrsList& coarseInterfaces_;

        //- Face-cell addressing
        labelList faceCellsHost_;
        labelgpuList faceCells_;
        labelgpuList sortCells_;
        labelgpuList cellFaces_;
        labelgpuList cellFacesStart_;		

        //- Face restrict addressing
        labelList faceRestrictAddressingHost_;
        labelgpuList faceRestrictSortAddressing_;
        labelgpuList faceRestrictTargetAddressing_;
        labelgpuList faceRestrictTargetStartAddressing_;

        void updateAddressing();

    // Protected Member Functions

        //- No copy construct
        GGAMGInterface(const GGAMGInterface&) = delete;

        //- No copy assignment
        void operator=(const GGAMGInterface&) = delete;


public:

    //- Runtime type information
    TypeName("GGAMGInterface");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            autoPtr,
            GGAMGInterface,
            gpulduInterface,
            (
                const label index,
                const gpulduInterfacePtrsList& coarseInterfaces,
                const gpulduInterface& fineInterface,
                const labelField& localRestrictAddressing,
                const labelField& neighbourRestrictAddressing,
                const label fineLevelIndex,
                const label coarseComm
            ),
            (
                index,
                coarseInterfaces,
                fineInterface,
                localRestrictAddressing,
                neighbourRestrictAddressing,
                fineLevelIndex,
                coarseComm
            )
        );

        declareRunTimeSelectionTable
        (
            autoPtr,
            GGAMGInterface,
            Istream,
            (
                const label index,
                const gpulduInterfacePtrsList& coarseInterfaces,
                Istream& is
            ),
            (
                index,
                coarseInterfaces,
                is
            )
        );


    // Selectors

        //- Return a pointer to a new interface created on freestore given
        //  the fine interface
        static autoPtr<GGAMGInterface> New
        (
            const label index,
            const gpulduInterfacePtrsList& coarseInterfaces,
            const gpulduInterface& fineInterface,
            const labelField& localRestrictAddressing,
            const labelField& neighbourRestrictAddressing,
            const label fineLevelIndex,
            const label coarseComm
        );

        //- Return a pointer to a new interface created on freestore given
        //  the fine interface
        static autoPtr<GGAMGInterface> New
        (
            const word& coupleType,
            const label index,
            const gpulduInterfacePtrsList& coarseInterfaces,
            Istream& is
        );


    // Constructors

        //- Construct from interfaces, restrict addressing set later on
        GGAMGInterface
        (
            const label index,
            const gpulduInterfacePtrsList& coarseInterfaces
        )
        :
            index_(index),
            coarseInterfaces_(coarseInterfaces)
        {}


        //- Construct from interfaces and restrict addressing
        GGAMGInterface
        (
            const label index,
            const gpulduInterfacePtrsList& coarseInterfaces,
            const labelUList& faceCells,
            const labelUList& faceRestrictAddressing
        )
        :
            index_(index),
            coarseInterfaces_(coarseInterfaces),
            faceCellsHost_(faceCells),
            faceCells_(faceCellsHost_),
            faceRestrictAddressingHost_(faceRestrictAddressing)
        {
            updateAddressing();
        }


        //- Construct from Istream
        GGAMGInterface
        (
            const label index,
            const gpulduInterfacePtrsList& coarseInterfaces,
            Istream& is
        );


    // Member Functions

        // Access

            //- Return size
            virtual label size() const
            {
                return faceCells_.size();
            }

            virtual label index() const
            {
                return index_;
            }

            virtual const gpulduInterfacePtrsList& coarseInterfaces() const
            {
                return coarseInterfaces_;
            }

            //- Return faceCell addressing
            virtual const labelUList& faceCells() const
            {
                return faceCellsHost_;
            }

            virtual const labelgpuList& sortCells() const
            {
                return sortCells_;
            }

            virtual const labelgpuList& cellFaces() const
            {
                return cellFaces_;
            }

            virtual const labelgpuList& cellFacesStart() const
            {
                return cellFacesStart_;
            }

            virtual const labelgpuList& gpuFaceCells() const
            {
                return faceCells_;
            }

            //- Return (local)face restrict addressing
            virtual const labelgpuList& faceRestrictSortAddressing() const
            {
                return faceRestrictSortAddressing_;
            }

            virtual const labelgpuList& faceRestrictTargetAddressing() const
            {
                return faceRestrictTargetAddressing_;
            }

            virtual const labelgpuList& faceRestrictTargetStartAddressing() const
            {
                return faceRestrictTargetStartAddressing_;
            }

            virtual const labelList& faceRestrictAddressingHost() const
            {
                return faceRestrictAddressingHost_;
            }

            //- Return non-const access to face restrict addressing
            /*
            virtual labelList& faceRestrictAddressing()
            {
                return faceRestrictAddressing_;
            }
            */

            //- Return the interface internal field of the given field
            template<class Type>
            tmp<Field<Type>> interfaceInternalField
            (
                const UList<Type>& internalData
            ) const;

            template<class Type>
            tmp<gpuField<Type>> interfaceInternalField
            (
                const gpuList<Type>& internalData
            ) const;

            //- Return the interface internal field of the given field
            //- using faceCell mapping
            template<class Type>
            tmp<Field<Type>> interfaceInternalField
            (
                const UList<Type>& internalData,
                const labelUList& faceCells
            ) const;

            //- Get the interface internal field of the given field
            template<class Type>
            void interfaceInternalField
            (
                const UList<Type>& internalData,
                List<Type>&
            ) const;


            template<class Type>
            void interfaceInternalField
            (
                const gpuList<Type>& internalData,
                gpuList<Type>&
            ) const;

            //- Return the values of the given internal data adjacent to
            //  the interface as a field
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData
            ) const;

            //- Return the values of the given internal data adjacent to
            //- the interface as a field using faceCell mapping
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData,
                const labelgpuList& faceCells
            ) const;


        // Agglomeration

            //- Merge the next level with this level
            //  combining the face-restrict addressing
            //  and copying the face-cell addressing
            void combine(const GGAMGInterface&);

            //- Agglomerating the given fine-level coefficients and return
            virtual tmp<scalargpuField> agglomerateCoeffs
            (
                const scalargpuField& fineCoeffs
            ) const;


        // I/O

            //- Write to stream
            virtual void write(Ostream&) const = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "GAMGInterfaceTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
