/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "GGAMGInterface.H"

#include "GAMGInterfaceF.H"
#include "GGAMGAgglomeration.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(GGAMGInterface, 0);
    defineRunTimeSelectionTable(GGAMGInterface, gpulduInterface);
    defineRunTimeSelectionTable(GGAMGInterface, Istream);
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::GGAMGInterface::GGAMGInterface
(
    const label index,
    const gpulduInterfacePtrsList& coarseInterfaces,
    Istream& is
)
:
    index_(index),
    coarseInterfaces_(coarseInterfaces),
    faceCellsHost_(is),
    faceCells_(faceCellsHost_),
    faceRestrictAddressingHost_(is)
{
    updateAddressing();
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::GGAMGInterface::updateAddressing()
{
    labelgpuList restrictTmp(faceRestrictAddressingHost_);

    GGAMGAgglomeration::createSort
    (
        restrictTmp,
        faceRestrictSortAddressing_
    );

    GGAMGAgglomeration::createTarget
    (
        restrictTmp,
        faceRestrictSortAddressing_,
        faceRestrictTargetAddressing_,
        faceRestrictTargetStartAddressing_
    );

    labelgpuList sort(faceCells_.size());
    GGAMGAgglomeration::createSort
    (
        faceCells_,
        cellFaces_
    );

    GGAMGAgglomeration::createTarget
    (
        faceCells_,
        cellFaces_,
        sortCells_,
        cellFacesStart_
    );
}

void Foam::GGAMGInterface::combine(const GGAMGInterface& coarseGi)
{
    const labelList& coarseFraHost = coarseGi.faceRestrictAddressingHost_;

    forAll(faceRestrictAddressingHost_, ffi)
    {
        faceRestrictAddressingHost_[ffi] = coarseFraHost[faceRestrictAddressingHost_[ffi]];
    }

    faceCells_ = coarseGi.faceCells_;
    faceCellsHost_ = coarseGi.faceCellsHost_;

    updateAddressing();
}


Foam::tmp<Foam::labelgpuField> Foam::GGAMGInterface::interfaceInternalField
(
    const labelgpuList& internalData
) const
{
    return interfaceInternalField<label>(internalData);
}


Foam::tmp<Foam::labelgpuField> Foam::GGAMGInterface::interfaceInternalField
(
    const labelgpuList& internalData,
    const labelgpuList& faceCells
) const
{
    //return interfaceInternalField<label>(internalData, faceCells);
    auto tfld = tmp<labelgpuField>::New(faceCells.size());
    auto& fld = tfld.ref();
    thrust::copy(thrust::make_permutation_iterator(internalData.begin(),faceCells.begin()),
                    thrust::make_permutation_iterator(internalData.begin(),faceCells.end()),
                    fld.begin());
    return tfld;
}


Foam::tmp<Foam::scalargpuField> Foam::GGAMGInterface::agglomerateCoeffs
(
    const scalargpuField& fineCoeffs
) const
{
    auto tcoarseCoeffs = tmp<scalargpuField>::New(size(), Zero);
    auto& coarseCoeffs = tcoarseCoeffs.ref();

    if (fineCoeffs.size() != faceRestrictAddressingHost_.size())
    {
        FatalErrorInFunction
            << "Size of coefficients " << fineCoeffs.size()
            << " does not correspond to the size of the restriction "
            << faceRestrictAddressingHost_.size()
            << abort(FatalError);
    }
    if (debug && max(faceRestrictAddressingHost_) > size())
    {
        FatalErrorInFunction
            << "Face restrict addressing addresses outside of coarse interface"
            << " size. Max addressing:" << max(faceRestrictAddressingHost_)
            << " coarse size:" << size()
            << abort(FatalError);
    }

    //forAll(faceRestrictAddressing_, ffi)
    //{
    //    coarseCoeffs[faceRestrictAddressing_[ffi]] += fineCoeffs[ffi];
    //}

    thrust::transform
    (
        faceRestrictTargetStartAddressing_.begin(),
        faceRestrictTargetStartAddressing_.end()-1,
        faceRestrictTargetStartAddressing_.begin()+1,
        thrust::make_permutation_iterator
        (
            coarseCoeffs.begin(),
            faceRestrictTargetAddressing_.begin()
        ),
        GAMGInterfaceAgglomerateCoeffs
        (
            fineCoeffs.data(),
            faceRestrictSortAddressing_.data()
        )
    );

    return tcoarseCoeffs;
}


void Foam::GGAMGInterface::write(Ostream& os) const
{
    os  << faceCellsHost_ << token::SPACE << faceRestrictAddressingHost_;
}


// ************************************************************************* //
