/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "PaSR.H"

namespace Foam
{

struct PaSRkappaFunctor 
{
    const scalar Cmix;

    PaSRkappaFunctor(const scalar Cmix_):Cmix(Cmix_){}

    __host__ __device__
    scalar operator()(const thrust::tuple<scalar,scalar,scalar,scalar>& t){
        const scalar tk =
                Cmix*sqrt(max(thrust::get<1>(t)/thrust::get<2>(t)/(thrust::get<0>(t) + SMALL), 0));
        if(tk > SMALL)
            return thrust::get<3>(t)/(thrust::get<0>(t) + tk);
        else
            return 1.0;
    } 
};
}

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class ReactionThermo>
Foam::combustionModels::PaSR<ReactionThermo>::PaSR
(
    const word& modelType,
    ReactionThermo& thermo,
    const compressibleTurbulenceModel& turb,
    const word& combustionProperties
)
:
    laminar<ReactionThermo>(modelType, thermo, turb, combustionProperties),
    Cmix_(this->coeffs().getScalar("Cmix")),
    kappa_
    (
        IOobject
        (
            thermo.phasePropertyName(typeName + ":kappa"),
            this->mesh().time().timeName(),
            this->mesh().hostmesh(),
            IOobject::NO_READ,
            IOobject::AUTO_WRITE
        ),
        this->mesh(),
        dimensionedScalar(dimless, Zero)
    )
{}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

template<class ReactionThermo>
Foam::combustionModels::PaSR<ReactionThermo>::~PaSR()
{}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //

template<class ReactionThermo>
void Foam::combustionModels::PaSR<ReactionThermo>::correct()
{
    if (this->active())
    {
        laminar<ReactionThermo>::correct();

        tmp<volScalargpuField> tepsilon(this->turbulence().epsilon());
        const scalargpuField& epsilon = tepsilon();

        tmp<volScalargpuField> tmuEff(this->turbulence().muEff());
        const scalargpuField& muEff = tmuEff();

        tmp<volScalargpuField> ttc(this->tc());
        const scalargpuField& tc = ttc();

        tmp<volScalargpuField> trho(this->rho());
        const scalargpuField& rho = trho();

        //forAll(epsilon, i)
        //{
        //    const scalar tk =
        //        Cmix_*sqrt(max(muEff[i]/rho[i]/(epsilon[i] + SMALL), 0));
//
        //    if (tk > SMALL)
        //    {
        //        kappa_[i] = tc[i]/(tc[i] + tk);
        //    }
        //    else
        //    {
        //        kappa_[i] = 1.0;
        //    }
        //}

        thrust::transform(thrust::make_zip_iterator(thrust::make_tuple(epsilon.begin(),
                                                    muEff.begin(),
                                                    rho.begin(),
                                                    tc.begin())),                        
                          thrust::make_zip_iterator(thrust::make_tuple(epsilon.end(),
                                                    muEff.end(),
                                                    rho.end(),
                                                    tc.end())),
                          kappa_.field().begin(),
                          PaSRkappaFunctor(Cmix_));

    }
}


template<class ReactionThermo>
Foam::tmp<Foam::gpufvScalarMatrix>
Foam::combustionModels::PaSR<ReactionThermo>::R(volScalargpuField& Y) const
{
    return kappa_*laminar<ReactionThermo>::R(Y);
}

template<class ReactionThermo>
Foam::tmp<Foam::volScalargpuField>
Foam::combustionModels::PaSR<ReactionThermo>::R2(volScalargpuField& Y) const 
{
    return kappa_*laminar<ReactionThermo>::R2(Y);
}

template<class ReactionThermo>
Foam::tmp<Foam::volScalargpuField>
Foam::combustionModels::PaSR<ReactionThermo>::Qdot() const
{
    return tmp<volScalargpuField>
    (
        new volScalargpuField
        (
            this->thermo().phasePropertyName(typeName + ":Qdot"),
            kappa_*laminar<ReactionThermo>::Qdot()
        )
    );
}


template<class ReactionThermo>
bool Foam::combustionModels::PaSR<ReactionThermo>::read()
{
    if (laminar<ReactionThermo>::read())
    {
        this->coeffs().readEntry("Cmix", Cmix_);
        return true;
    }

    return false;
}


// ************************************************************************* //
