/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016, 2019 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "omegaWallFunctionFvPatchScalargpuField.H"
#include "nutWallFunctionFvPatchScalargpuField.H"
#include "turbulenceModel.H"
#include "gpufvMatrix.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

const Foam::Enum
<
    Foam::omegaWallFunctionFvPatchScalargpuField::blendingType
>
Foam::omegaWallFunctionFvPatchScalargpuField::blendingTypeNames
({
    { blendingType::STEPWISE , "stepwise" },
    { blendingType::MAX , "max" },
    { blendingType::BINOMIAL2 , "binomial2" },
    { blendingType::BINOMIAL , "binomial" },
    { blendingType::EXPONENTIAL, "exponential" },
    { blendingType::TANH, "tanh" }
});

Foam::scalar Foam::omegaWallFunctionFvPatchScalargpuField::tolerance_ = 1e-5;

// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //
namespace Foam
{
struct omegaWallFunctionFvPatchScalarFieldCreateWeightsFunctor: public std::unary_function<thrust::tuple<label,label>,label>
{
    __host__ __device__
    label operator()(const thrust::tuple<label,label>& t)
    {
        return thrust::get<1>(t) - thrust::get<0>(t);
    }
};
}

void Foam::omegaWallFunctionFvPatchScalargpuField::setMaster()
{
    if (master_ != -1)
    {
        return;
    }

    const volScalargpuField& omega =
        static_cast<const volScalargpuField&>(this->internalField());

    const volScalargpuField::Boundary& bf = omega.boundaryField();

    label master = -1;
    forAll(bf, patchi)
    {
        if (isA<omegaWallFunctionFvPatchScalargpuField>(bf[patchi]))
        {
            omegaWallFunctionFvPatchScalargpuField& opf = omegaPatch(patchi);

            if (master == -1)
            {
                master = patchi;
            }

            opf.master() = master;
        }
    }
}


void Foam::omegaWallFunctionFvPatchScalargpuField::createAveragingWeights()
{
    const volScalargpuField& omega =
        static_cast<const volScalargpuField&>(this->internalField());

    const volScalargpuField::Boundary& bf = omega.boundaryField();

    const gpufvMesh& mesh = omega.mesh();

    if (initialised_ && !mesh.hostmesh().changing())
    {
        return;
    }

    volScalargpuField weights
    (
        IOobject
        (
            "weights",
            mesh.time().timeName(),
            mesh.hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false // do not register
        ),
        mesh,
        dimensionedScalar(dimless, Zero)
    );

    DynamicList<label> omegaPatches(bf.size());
    forAll(bf, patchi)
    {
        if (isA<omegaWallFunctionFvPatchScalargpuField>(bf[patchi]))
        {
            omegaPatches.append(patchi);

            const labelgpuList& pcells = mesh.lduAddr().gpuPatchSortCells(patchi);
            const labelgpuList& losortStart = mesh.lduAddr().gpuPatchSortStartAddr(patchi);
            
            thrust::transform
            (
                thrust::make_permutation_iterator
                (
                    weights.begin(),
                    pcells.begin()
                ),
                thrust::make_permutation_iterator
                (
                    weights.begin(),
                    pcells.end()
                ),
                thrust::make_transform_iterator
                (
                    thrust::make_zip_iterator(thrust::make_tuple
                    (
                        losortStart.begin(),
                        losortStart.begin()+1
                    )),
                    omegaWallFunctionFvPatchScalarFieldCreateWeightsFunctor()
                ),
                thrust::make_permutation_iterator
                (
                    weights.begin(),pcells.begin()
                ),
                addOperatorFunctor<scalar,label,scalar>()
            );
        }
    }

    cornerWeights_.setSize(bf.size());
    for (const auto& patchi : omegaPatches)
    {
        const fvPatchScalargpuField& wf = weights.boundaryField()[patchi];
        cornerWeights_[patchi] = 1.0/wf.patchInternalField();
    }

    G_.setSize(internalField().size(), 0.0);
    omega_.setSize(internalField().size(), 0.0);

    initialised_ = true;
}


Foam::omegaWallFunctionFvPatchScalargpuField&
Foam::omegaWallFunctionFvPatchScalargpuField::omegaPatch
(
    const label patchi
)
{
    const volScalargpuField& omega =
        static_cast<const volScalargpuField&>(this->internalField());

    const volScalargpuField::Boundary& bf = omega.boundaryField();

    const omegaWallFunctionFvPatchScalargpuField& opf =
        refCast<const omegaWallFunctionFvPatchScalargpuField>(bf[patchi]);

    return const_cast<omegaWallFunctionFvPatchScalargpuField&>(opf);
}


void Foam::omegaWallFunctionFvPatchScalargpuField::calculateTurbulenceFields
(
    const turbulenceModel& turbModel,
    scalargpuField& G0,
    scalargpuField& omega0
)
{
    // accumulate all of the G and omega contributions
    forAll(cornerWeights_, patchi)
    {
        if (!cornerWeights_[patchi].empty())
        {
            omegaWallFunctionFvPatchScalargpuField& opf = omegaPatch(patchi);

            const gpuList<scalar>& w = cornerWeights_[patchi];

            opf.calculate(turbModel, w, opf.patch(), G0, omega0);
        }
    }

    // apply zero-gradient condition for omega
    forAll(cornerWeights_, patchi)
    {
        if (!cornerWeights_[patchi].empty())
        {
            omegaWallFunctionFvPatchScalargpuField& opf = omegaPatch(patchi);

            opf == scalargpuField(omega0, opf.patch().gpuFaceCells());
        }
    }
}

namespace Foam
{
struct omegaCalculateOmegaFunctor : public std::unary_function<label,scalar>
{
    const scalar Cmu25;
    const scalar beta1;
    const scalar kappa;
    const scalar yPlusLam;
    const scalar n;
    const label blend;
	const scalar* nuw;
	const scalar* cornerWeights;
	const scalar* y;
	const scalar* k;
	
    omegaCalculateOmegaFunctor
	(
        const scalar Cmu25_,
		const scalar beta1_,
        const scalar kappa_,
        const scalar yPlusLam_,
        const scalar n_,
        const label blend_,
        const scalar* nuw_,
		const scalar* cornerWeights_,
   
		const scalar* y_,
		const scalar* k_
	):
        Cmu25(Cmu25_),
		beta1(beta1_),
        kappa(kappa_),
        yPlusLam(yPlusLam_),
        n(n_),
        blend(blend_),
        nuw(nuw_),
        cornerWeights(cornerWeights_),
        y(y_),
        k(k_)
	{}

	__host__ __device__
   
	scalar operator()(const label& cellI,const label& faceI)
	{
        
		const scalar yPlus = Cmu25*y[faceI]*sqrt(k[cellI])/nuw[faceI];
        const scalar w = cornerWeights[faceI];
		
        const scalar omegaVis = 6.0*nuw[faceI]/(beta1*sqr(y[faceI]));
		
        const scalar omegaLog = sqrt(k[cellI])/(Cmu25*kappa*y[faceI]);

        switch(blend)
        {
            case 0:
            {
                if (yPlus > yPlusLam)
                {
                    return w*omegaLog;
                }
                else
                {
                    return w*omegaVis;
                }
            }
                
            case 1:
            {
                return max(omegaVis, omegaLog);
            }

            case 2:
            {
                return w*sqrt(sqr(omegaVis) + sqr(omegaLog));
            }
 
            case 3:
            {
                return pow
                (
                    pow(omegaVis, n) + pow(omegaLog, n),
                    1.0/n
                );
            }
 
            case 4:
			{
                const scalar Gamma = 0.01*pow4(yPlus)/(1.0 + 5.0*yPlus);
                const scalar invGamma = 1.0/(Gamma + ROOTVSMALL);
                return w*(omegaVis*exp(-Gamma) + omegaLog*exp(-invGamma));
            }
			
            case 5:
            {
                // (KAS:Eqs. 33-34)
                const scalar phiTanh = tanh(pow4(yPlus/10.0));
                const scalar omegab1 = omegaVis + omegaLog;
                const scalar omegab2 =
                    pow(pow(omegaVis, 1.2) + pow(omegaLog, 1.2), 1.0/1.2);

                return phiTanh*omegab1 + (1.0 - phiTanh)*omegab2;
            }
       	}
	}
};
	
struct omegaCalculateGFunctor : public std::unary_function<label,scalar>
{
    const scalar Cmu25;
    const scalar kappa;
    const scalar yPlusLam;
	const label blend;
    const scalar* cornerWeights;
    const scalar* y;
    const scalar* k;
    const scalar* nuw;
    const scalar* nutw;
    const scalar* magGradUw;

    omegaCalculateGFunctor
    (
        const scalar Cmu25_,
        const scalar kappa_,
        const scalar yPlusLam_,
        const label blend_,
        const scalar* cornerWeights_,
        const scalar* y_,
        const scalar* k_,
        const scalar* nuw_,
        const scalar* nutw_,
        const scalar* magGradUw_
    ):
        Cmu25(Cmu25_),
        kappa(kappa_),
        yPlusLam(yPlusLam_),
		blend(blend_),
        cornerWeights(cornerWeights_),
        y(y_),
        k(k_),
        nuw(nuw_),
        nutw(nutw_),
        magGradUw(magGradUw_)
        {}

    __host__ __device__
    scalar operator()(const label& cellI,const label& faceI)
    {
		const scalar yPlus = Cmu25*y[faceI]*sqrt(k[cellI])/nuw[faceI];
        const scalar w = cornerWeights[faceI];

        if (!(blend == 0) || yPlus > yPlusLam)
        {
        return
            w
           *(nutw[faceI] + nuw[faceI])
           *magGradUw[faceI]
           *Cmu25*sqrt(k[cellI])
           /(kappa*y[faceI]);
        }
		else
		{
		return 0.0;
		}
    }
};
}
	

void Foam::omegaWallFunctionFvPatchScalargpuField::calculate
(
    const turbulenceModel& turbModel,
    const gpuList<scalar>& cornerWeights,
    const gpufvPatch& patch,
    scalargpuField& G0,
    scalargpuField& omega0
)
{
    const label patchi = patch.index();

    const nutWallFunctionFvPatchScalargpuField& nutw =
        nutWallFunctionFvPatchScalargpuField::nutw(turbModel, patchi);

    const scalargpuField& y = turbModel.y()[patchi];

    const tmp<scalargpuField> tnuw = turbModel.nu(patchi);
    const scalargpuField& nuw = tnuw();

    const tmp<volScalargpuField> tk = turbModel.k();
    const volScalargpuField& k = tk();

    const fvPatchVectorgpuField& Uw = turbModel.U().boundaryField()[patchi];

    const scalargpuField magGradUw(mag(Uw.snGrad()));

    const scalar Cmu25 = pow025(nutw.Cmu());
	
    const scalar kappa = nutw.kappa();
    const scalar yPlusLam = nutw.yPlusLam();

    label blending = 0;
   
    switch (blending_)
    {
        case blendingType::STEPWISE:
        {
            blending = 0;
            break;
        }

        case blendingType::MAX:
        {
            blending = 1;
            break;
        }

        case blendingType::BINOMIAL2:
        {
            blending = 2;
            break;
        }
		
        case blendingType::BINOMIAL:
        {
            blending = 3;
            break;
        }

        case blendingType::EXPONENTIAL:
        {
            blending = 4;
            break;
        }
		
        case blendingType::TANH:
        {
            blending = 5;
            break;
        }
    }
	
    matrixPatchOperation
    (
        patchi,
        omega0,
        patch.boundaryMesh().mesh().lduAddr(),
        omegaCalculateOmegaFunctor
        (
            Cmu25,
            beta1_,
            kappa,
            yPlusLam,
            n_,
            blending,
            nuw.data(),
            cornerWeights.data(),
            y.data(),
            k.data()
        )
    );

    matrixPatchOperation
    (
        patchi,
        G0,
        patch.boundaryMesh().mesh().lduAddr(),
        omegaCalculateGFunctor
        (
            Cmu25,
            kappa,
            yPlusLam,
            blending,
            cornerWeights.data(),
            y.data(),
            k.data(),
            nuw.data(),
            nutw.data(),
            magGradUw.data()
        )
    );
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::omegaWallFunctionFvPatchScalargpuField::omegaWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedValueFvPatchgpuField<scalar>(p, iF),
    blending_(blendingType::BINOMIAL2),
    n_(2.0),
    initialised_(false),
    master_(-1),
    beta1_(0.075),
    G_(),
    omega_(),
    cornerWeights_()
{}


Foam::omegaWallFunctionFvPatchScalargpuField::omegaWallFunctionFvPatchScalargpuField
(
    const omegaWallFunctionFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    fixedValueFvPatchgpuField<scalar>(ptf, p, iF, mapper),
    blending_(ptf.blending_),
    n_(ptf.n_),
    initialised_(false),
    master_(-1),
    beta1_(ptf.beta1_),
    G_(),
    omega_(),
    cornerWeights_()
{}


Foam::omegaWallFunctionFvPatchScalargpuField::omegaWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    fixedValueFvPatchgpuField<scalar>(p, iF, dict),
    blending_
    (
        blendingTypeNames.getOrDefault
        (
            "blending",
            dict,
            blendingType::BINOMIAL2
        )
    ),
    n_
    (
        dict.getCheckOrDefault<scalar>
        (
            "n",
            2.0,
            scalarMinMax::ge(0)
        )
    ),
    initialised_(false),
    master_(-1),
    beta1_(dict.getOrDefault<scalar>("beta1", 0.075)),
    G_(),
    omega_(),
    cornerWeights_()
{
    // The deprecated 'blended' keyword is superseded by the enum 'blending'
    if (dict.found("blended"))
    {
        IOWarningInFunction(dict)
            << "Using deprecated 'blended' keyword"
            << nl << "    Please use either of the below for the same behaviour:"
            << nl << "    'blending  binomial2;' for 'blended  on;'"
            << nl << "    'blending  stepwise;'  for 'blended  off;'"
            << nl << "    OVERWRITING: 'blended' keyword -> 'blending' enum"
            << endl;

        bool blended = dict.get<bool>("blended");

        if (blended)
        {
            blending_ = blendingType::BINOMIAL2;
        }
        else
        {
            blending_ = blendingType::STEPWISE;
        }
    }

    // apply zero-gradient condition on start-up
    this->operator==(patchInternalField());
}


Foam::omegaWallFunctionFvPatchScalargpuField::omegaWallFunctionFvPatchScalargpuField
(
    const omegaWallFunctionFvPatchScalargpuField& owfpsf
)
:
    fixedValueFvPatchgpuField<scalar>(owfpsf),
    blending_(owfpsf.blending_),
    n_(owfpsf.n_),
    initialised_(false),
    master_(-1),
    beta1_(owfpsf.beta1_),
    G_(),
    omega_(),
    cornerWeights_()
{}


Foam::omegaWallFunctionFvPatchScalargpuField::omegaWallFunctionFvPatchScalargpuField
(
    const omegaWallFunctionFvPatchScalargpuField& owfpsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedValueFvPatchgpuField<scalar>(owfpsf, iF),
    blending_(owfpsf.blending_),
    n_(owfpsf.n_),
    initialised_(false),
    master_(-1),
    beta1_(owfpsf.beta1_),
    G_(),
    omega_(),
    cornerWeights_()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::scalargpuField& Foam::omegaWallFunctionFvPatchScalargpuField::G
(
    bool init
)
{
    if (patch().index() == master_)
    {
        if (init)
        {
            G_ = 0.0;
        }

        return G_;
    }

    return omegaPatch(master_).G();
}


Foam::scalargpuField& Foam::omegaWallFunctionFvPatchScalargpuField::omega
(
    bool init
)
{
    if (patch().index() == master_)
    {
        if (init)
        {
            omega_ = 0.0;
        }

        return omega_;
    }

    return omegaPatch(master_).omega(init);
}


void Foam::omegaWallFunctionFvPatchScalargpuField::updateCoeffs()
{
    if (updated())
    {
        return;
    }

    const turbulenceModel& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );

    setMaster();

    if (patch().index() == master_)
    {
        createAveragingWeights();
        calculateTurbulenceFields(turbModel, G(true), omega(true));
    }

    const scalargpuField& G0 = this->G();
    const scalargpuField& omega0 = this->omega();

    typedef DimensionedgpuField<scalar, gpuvolMesh> FieldType;

    FieldType& G = db().lookupObjectRef<FieldType>(turbModel.GName());

    FieldType& omega = const_cast<FieldType&>(internalField());

    thrust::copy(thrust::make_permutation_iterator(G0.begin(),patch().gpuFaceCells().begin()),
                 thrust::make_permutation_iterator(G0.begin(),patch().gpuFaceCells().end()),
                 thrust::make_permutation_iterator(G.begin(),patch().gpuFaceCells().begin()));
                 
    thrust::copy(thrust::make_permutation_iterator(omega0.begin(),patch().gpuFaceCells().begin()),
                 thrust::make_permutation_iterator(omega0.begin(),patch().gpuFaceCells().end()),
                 thrust::make_permutation_iterator(omega.begin(),patch().gpuFaceCells().begin()));

    fvPatchgpuField<scalar>::updateCoeffs();
}

namespace Foam
{
struct omegaWallFunctionFvPatchScalarFieldupdateCoeffsFunctor
{
	const scalar tolerance_;
	omegaWallFunctionFvPatchScalarFieldupdateCoeffsFunctor(scalar tolerance):
		tolerance_(tolerance){}
    __host__ __device__
    scalar operator () (const scalar& w,const thrust::tuple<scalar,scalar>& t)
    {
        scalar G = thrust::get<0>(t);
        scalar G0 = thrust::get<1>(t);
		
        if(w>tolerance_)return (1.0-w)*G+w*G0;
    }	
	
};
}

void Foam::omegaWallFunctionFvPatchScalargpuField::updateWeightedCoeffs
(
    const scalargpuField& weights
)
{
    if (updated())
    {
        return;
    }

    const turbulenceModel& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );

    setMaster();

    if (patch().index() == master_)
    {
        createAveragingWeights();
        calculateTurbulenceFields(turbModel, G(true), omega(true));
    }

    const scalargpuField& G0 = this->G();
    const scalargpuField& omega0 = this->omega();

    typedef DimensionedgpuField<scalar, gpuvolMesh> FieldType;

    FieldType& G = db().lookupObjectRef<FieldType>(turbModel.GName());

    FieldType& omega = const_cast<FieldType&>(internalField());

    scalargpuField& omegaf = *this;

    // only set the values if the weights are > tolerance
    thrust::transform
    (
        weights.begin(),
        weights.end(),
        thrust::make_zip_iterator(thrust::make_tuple
        (
            thrust::make_permutation_iterator(G.begin(),  patch().gpuFaceCells().begin()),
            thrust::make_permutation_iterator(G0.begin(),  patch().gpuFaceCells().begin())
        )),
        thrust::make_permutation_iterator(G.begin(),patch().gpuFaceCells().begin()),
        omegaWallFunctionFvPatchScalarFieldupdateCoeffsFunctor(tolerance_)
    );
                     
    thrust::transform
    (
        weights.begin(),
        weights.end(),
        thrust::make_zip_iterator(thrust::make_tuple
        (
            thrust::make_permutation_iterator(omega.begin(),  patch().gpuFaceCells().begin()),
            thrust::make_permutation_iterator(omega0.begin(),  patch().gpuFaceCells().begin())
        )),
        thrust::make_permutation_iterator(omega.begin(),patch().gpuFaceCells().begin()),
        omegaWallFunctionFvPatchScalarFieldupdateCoeffsFunctor(tolerance_)
    );
                     
    thrust::copy
    (
        thrust::make_permutation_iterator(omega.begin(),patch().gpuFaceCells().begin()),
        thrust::make_permutation_iterator(omega.begin(),patch().gpuFaceCells().end()),
        omegaf.begin()
    );

    fvPatchgpuField<scalar>::updateCoeffs();
}

namespace Foam
{
struct omegaWallFunctionGraterThanToleranceFunctor : public std::unary_function<scalar,bool>{
	const scalar tolerance_;
	omegaWallFunctionGraterThanToleranceFunctor(scalar tolerance): tolerance_(tolerance){}
    __host__ __device__
	bool operator () (const scalar& s){
		return s > tolerance_;
	}
};
}

void Foam::omegaWallFunctionFvPatchScalargpuField::manipulateMatrix
(
    gpufvMatrix<scalar>& matrix
)
{
    if (manipulatedMatrix())
    {
        return;
    }

    matrix.setValues(patch().gpuFaceCells(), patchInternalField());

    fvPatchgpuField<scalar>::manipulateMatrix(matrix);
}


void Foam::omegaWallFunctionFvPatchScalargpuField::manipulateMatrix
(
    gpufvMatrix<scalar>& matrix,
    const gpuField<scalar>& weights
)
{
    if (manipulatedMatrix())
    {
        return;
    }

    gpuList<label> constraintCells(weights.size());
    gpuList<scalar> constraintValues(weights.size());
    const labelgpuList& faceCells = patch().gpuFaceCells();

    const DimensionedgpuField<scalar, gpuvolMesh>& fld = internalField();

    typename gpuList<label>::iterator end = 
        thrust::copy_if
        (
            faceCells.begin(),
            faceCells.end(),
            weights.begin(),
            constraintCells.begin(),
            omegaWallFunctionGraterThanToleranceFunctor(tolerance_)
        );
                               
    label nConstrainedCells = end - constraintCells.begin();
    
    constraintCells.setSize(nConstrainedCells);
    
    thrust::copy_if
    (
        thrust::make_permutation_iterator(fld.begin(),faceCells.begin()),
        thrust::make_permutation_iterator(fld.begin(),faceCells.end()),
        weights.begin(),
        constraintValues.begin(),
        omegaWallFunctionGraterThanToleranceFunctor(tolerance_)
    );
                               
    constraintValues.setSize(nConstrainedCells);

    if (debug)
    {
        Pout<< "Patch: " << patch().name()
            << ": number of constrained cells = " << constraintCells.size()
            << " out of " << patch().size()
            << endl;
    }

    thrust::sort_by_key
    (
        constraintCells.begin(),
        constraintCells.end(),
        constraintValues.begin()
    );

    matrix.setValues(constraintCells, constraintValues);

    fvPatchgpuField<scalar>::manipulateMatrix(matrix);
}


void Foam::omegaWallFunctionFvPatchScalargpuField::write
(
    Ostream& os
) const
{
    os.writeEntry("blending", blendingTypeNames[blending_]);
    os.writeEntry("n", n_);
    os.writeEntry("beta1", beta1_);
    fixedValueFvPatchgpuField<scalar>::write(os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        omegaWallFunctionFvPatchScalargpuField
    );
}


// ************************************************************************* //
