/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016, 2019 OpenFOAM Foundation
    Copyright (C) 2019-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::nutkRoughWallFunctionFvPatchScalarField

Group
    grpWallFunctions

Description
    This boundary condition provides a wall constraint on the turbulent
    viscosity, i.e. \c nut, when using wall functions for rough walls,
    based on the turbulent kinetic energy, i.e. \c k. The condition
    manipulates the wall roughness parameter, i.e. \c E, to account
    for roughness effects.

    Parameter ranges:
    - roughness height = sand-grain roughness (0 for smooth walls)
    - roughness constant = 0.5-1.0

Usage
    Example of the boundary condition specification:
    \verbatim
    <patchName>
    {
        // Mandatory entries (unmodifiable)
        type            nutkRoughWallFunction;
        Ks              uniform 0;
        Cs              uniform 0.5;

        // Optional (inherited) entries
        ...
    }
    \endverbatim

    where the entries mean:
    \table
      Property    | Description                       | Type   | Req'd  | Dflt
      type        | Type name: nutkRoughWallFunction  | word   | yes    | -
      Ks          | Sand-grain roughness height  | scalarField | yes    | -
      Cs          | Roughness constant           | scalarField | yes    | -
    \endtable

    The inherited entries are elaborated in:
      - \link nutkWallFunctionFvPatchScalarField.H \endlink

SourceFiles
    nutkRoughWallFunctionFvPatchScalarField.C

\*---------------------------------------------------------------------------*/

#ifndef nutkRoughWallFunctionFvPatchScalargpuField_H
#define nutkRoughWallFunctionFvPatchScalargpuField_H

#include "nutkWallFunctionFvPatchScalargpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
           Class nutkRoughWallFunctionFvPatchScalarField Declaration
\*---------------------------------------------------------------------------*/

class nutkRoughWallFunctionFvPatchScalargpuField
:
    public nutkWallFunctionFvPatchScalargpuField
{
protected:

    // Protected Data

        //- Roughness height
        scalargpuField Ks_;

        //- Roughness constant
        scalargpuField Cs_;


    // Protected Member Functions

        //- Calculate the turbulent viscosity
        virtual tmp<scalargpuField> calcNut() const;


public:

    //- Runtime type information
    TypeName("nutkRoughWallFunction");


    // Constructors

        //- Construct from patch and internal field
        nutkRoughWallFunctionFvPatchScalargpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        nutkRoughWallFunctionFvPatchScalargpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping given
        //- nutkRoughWallFunctionFvPatchScalarField
        //- onto a new patch
        nutkRoughWallFunctionFvPatchScalargpuField
        (
            const nutkRoughWallFunctionFvPatchScalargpuField&,
            const gpufvPatch&,
            const DimensionedgpuField<scalar, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        nutkRoughWallFunctionFvPatchScalargpuField
        (
            const nutkRoughWallFunctionFvPatchScalargpuField&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchScalargpuField> clone() const
        {
            return tmp<fvPatchScalargpuField>
            (
                new nutkRoughWallFunctionFvPatchScalargpuField(*this)
            );
        }

        //- Construct as copy setting internal field reference
        nutkRoughWallFunctionFvPatchScalargpuField
        (
            const nutkRoughWallFunctionFvPatchScalargpuField&,
            const DimensionedgpuField<scalar, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchScalargpuField> clone
        (
            const DimensionedgpuField<scalar, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchScalargpuField>
            (
                new nutkRoughWallFunctionFvPatchScalargpuField(*this, iF)
            );
        }


    // Member Functions

        // Access functions

            // Return Ks
            scalargpuField& Ks()
            {
                return Ks_;
            }

            // Return Cs
            scalargpuField& Cs()
            {
                return Cs_;
            }


        // Mapping functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap(const fvPatchgpuFieldMapper&);

            //- Reverse map the given fvPatchField onto this fvPatchField
            virtual void rmap
            (
                const fvPatchScalargpuField&,
                const labelgpuList&
            );


        // I-O

            //- Write
            virtual void write(Ostream&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
