/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2015 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::compressibleTurbulenceModel

Description
    Abstract base class for turbulence models (RAS, LES and laminar).

SourceFiles
    compressibleTurbulenceModel.C

\*---------------------------------------------------------------------------*/

#ifndef compressibleTurbulenceModel_H
#define compressibleTurbulenceModel_H

#include "turbulenceModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declarations
class gpufvMesh;

/*---------------------------------------------------------------------------*\
                Class compressibleTurbulenceModel Declaration
\*---------------------------------------------------------------------------*/

class compressibleTurbulenceModel
:
    public turbulenceModel
{

protected:

    // Protected data

        const volScalargpuField& rho_;


    // Protected member functions

        //- ***HGW Temporary function to be removed when the run-time selectable
        //  thermal transport layer is complete
        virtual void correctNut()
        {}


private:

    // Private Member Functions

        //- No copy construct
        compressibleTurbulenceModel
        (
            const compressibleTurbulenceModel&
        ) = delete;

        //- No copy assignment
        void operator=(const compressibleTurbulenceModel&) = delete;


public:

    //- Runtime type information
    TypeName("compressibleTurbulenceModel");


    // Constructors

        //- Construct from components
        compressibleTurbulenceModel
        (
            const volScalargpuField& rho,
            const volVectorgpuField& U,
            const surfaceScalargpuField& alphaRhoPhi,
            const surfaceScalargpuField& phi,
            const word& propertiesName
        );


    //- Destructor
    virtual ~compressibleTurbulenceModel() = default;


    // Member functions

        //- Return the density field
        const volScalargpuField& rho() const
        {
            return rho_;
        }

        //- Return the volumetric flux field
        virtual tmp<surfaceScalargpuField> phi() const;

        //- Return the effective stress tensor including the laminar stress
        virtual tmp<volSymmTensorgpuField> devRhoReff() const = 0;

        //- Return the effective stress tensor including
        //- the laminar stress based on a given velocity field
        virtual tmp<volSymmTensorgpuField> devRhoReff
        (
            const volVectorgpuField& U
        ) const = 0;

        //- Return the source term for the momentum equation
        virtual tmp<gpufvVectorMatrix> divDevRhoReff(volVectorgpuField& U) const = 0;

        //- Correct the turbulence thermal diffusivity for energy transport
        virtual void correctEnergyTransport();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
