/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::SymmTensor

Description
    A templated (3 x 3) symmetric tensor of objects of \<T\>, effectively
    containing 6 elements, derived from VectorSpace.

See also
    Test-SymmTensor.C

SourceFiles
    SymmTensorI.H

\*---------------------------------------------------------------------------*/

#ifndef SymmTensor_H
#define SymmTensor_H

#include "contiguous.H"
#include "Vector.H"
#include "SphericalTensor.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class SymmTensor Declaration
\*---------------------------------------------------------------------------*/

template<class Cmpt>
class SymmTensor
:
    public VectorSpace<SymmTensor<Cmpt>, Cmpt, 6>
{
public:

    // Typedefs

        //- Equivalent type of labels used for valid component indexing
        typedef SymmTensor<label> labelType;


    // Member Constants

        //- Rank of SymmTensor is 2
        static constexpr direction rank = 2;


    // Static Data Members

        static const SymmTensor I;


    //- Component labeling enumeration
    enum components { XX, XY, XZ, YY, YZ, ZZ };


    // Generated Methods

        //- Default construct
        __host__ __device__
        SymmTensor() = default;

        //- Copy construct
        __host__ __device__
        SymmTensor(const SymmTensor&) = default;

        //- Copy assignment
        __host__ __device__
        SymmTensor& operator=(const SymmTensor&) = default;


    // Constructors

        //- Construct initialized to zero
        __host__ __device__
        inline SymmTensor(const Foam::zero);

        //- Construct given VectorSpace of the same rank
        template<class Cmpt2>
        __host__ __device__
        inline SymmTensor(const VectorSpace<SymmTensor<Cmpt2>, Cmpt2, 6>&);

        //- Construct given SphericalTensor
        __host__ __device__
        inline SymmTensor(const SphericalTensor<Cmpt>&);

        //- Construct given the six components
        __host__ __device__
        inline SymmTensor
        (
            const Cmpt txx, const Cmpt txy, const Cmpt txz,
                            const Cmpt tyy, const Cmpt tyz,
                                            const Cmpt tzz
        );

        //- Construct from Istream
        inline explicit SymmTensor(Istream& is);


    // Member Functions

        // Component access

            __host__ __device__
            inline const Cmpt& xx() const;
            __host__ __device__
            inline const Cmpt& xy() const;
            __host__ __device__
            inline const Cmpt& xz() const;
            __host__ __device__
            inline const Cmpt& yx() const;
            __host__ __device__
            inline const Cmpt& yy() const;
            __host__ __device__
            inline const Cmpt& yz() const;
            __host__ __device__
            inline const Cmpt& zx() const;
            __host__ __device__
            inline const Cmpt& zy() const;
            __host__ __device__
            inline const Cmpt& zz() const;

            __host__ __device__
            inline Cmpt& xx();
            __host__ __device__
            inline Cmpt& xy();
            __host__ __device__
            inline Cmpt& xz();
            __host__ __device__
            inline Cmpt& yx();
            __host__ __device__
            inline Cmpt& yy();
            __host__ __device__
            inline Cmpt& yz();
            __host__ __device__
            inline Cmpt& zx();
            __host__ __device__
            inline Cmpt& zy();
            __host__ __device__
            inline Cmpt& zz();


        // Diagonal access and manipulation

            //- Extract the diagonal as a vector
            __host__ __device__
            inline Vector<Cmpt> diag() const;

            //- Set values of the diagonal
            __host__ __device__
            inline void diag(const Vector<Cmpt>& v);


    // Tensor Operations

        //- Return non-Hermitian transpose
        __host__ __device__
        inline const SymmTensor<Cmpt>& T() const;


    // Member Operators

        //- Inherit VectorSpace assignment operators
        using SymmTensor::vsType::operator=;

        //- Assign to given SphericalTensor
        __host__ __device__
        inline void operator=(const SphericalTensor<Cmpt>&);
};


// * * * * * * * * * * * * * * * * * Traits  * * * * * * * * * * * * * * * * //

//- Data are contiguous if component type is contiguous
template<class Cmpt>
struct is_contiguous<SymmTensor<Cmpt>> : is_contiguous<Cmpt> {};

//- Data are contiguous label if component type is label
template<class Cmpt>
struct is_contiguous_label<SymmTensor<Cmpt>> : is_contiguous_label<Cmpt> {};

//- Data are contiguous scalar if component type is scalar
template<class Cmpt>
struct is_contiguous_scalar<SymmTensor<Cmpt>> : is_contiguous_scalar<Cmpt> {};


template<class Cmpt>
class symmTypeOfRank<Cmpt, 2>
{
public:

    typedef SymmTensor<Cmpt> type;
};


template<class Cmpt>
class typeOfSolve<SymmTensor<Cmpt>>
{
public:

    typedef SymmTensor<solveScalar> type;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "SymmTensorI.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
