/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::distributionModels::exponential

Description
    Particle-size distribution model wherein random samples are drawn
    from the doubly-truncated exponential probability density function:

    \f[
        f(x; \lambda, A, B) =
            \lambda \frac{\exp(-\lambda (x - A))}{1 - \exp(-\lambda(B-A))}
    \f]
    where

    \vartable
        f(x; \lambda, A, B) | Exponential probability density function
        \lambda   | Rate parameter
        x         | Sample
        A         | Minimum of the distribution
        B         | Maximum of the distribution
    \endvartable

    Constraints:
    - \f$ \infty > B > A > 0 \f$
    - \f$ x \in [B,A] \f$
    - \f$ \lambda > 0 \f$

    Random samples are generated by the inverse transform sampling technique
    by using the quantile function of the doubly-truncated exponential
    probability density function:

    \f[
        x = - \frac{1}{\lambda} \ln (r)
    \f]

    with

    \f[
        r = q_{min} + u (q_{max} - q_{min})
    \f]

    \f[
        q_{min} = \exp(-\lambda A)
    \f]

    \f[
        q_{max} = \exp(-\lambda B)
    \f]
    where \f$ u \f$ is a sample drawn from the uniform probability
    density function on the unit interval \f$ (0, 1) \f$.

Usage
    Minimal example by using \c constant/\<CloudProperties\>:
    \verbatim
    subModels
    {
        injectionModels
        {
            <name>
            {
                ...

                sizeDistribution
                {
                    type        exponential;
                    exponentialDistribution
                    {
                        lambda      <lambdaValue>;
                        minValue    <minValue>;
                        maxValue    <maxValue>;
                    }
                }
            }
        }
    }
    \endverbatim

    where the entries mean:
    \table
      Property     | Description                       | Type | Reqd | Deflt
      type         | Type name: exponential            | word | yes  | -
      exponentialDistribution | Distribution settings  | dict | yes  | -
      lambda       | Rate parameter                    | scalar | yes  | -
      minValue     | Minimum of the distribution       | scalar | yes  | -
      maxValue     | Maximum of the distribution       | scalar | yes  | -
    \endtable

SourceFiles
    exponential.C

\*---------------------------------------------------------------------------*/

#ifndef distributionModels_exponential_H
#define distributionModels_exponential_H

#include "distributionModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace distributionModels
{

/*---------------------------------------------------------------------------*\
                        Class exponential Declaration
\*---------------------------------------------------------------------------*/

class exponential
:
    public distributionModel
{
    // Private Data

        //- Rate parameter
        scalar lambda_;


public:

    //- Runtime type information
    TypeName("exponential");


    // Constructors

        //- Construct from components
        exponential(const dictionary& dict, Random& rndGen);

        //- Copy construct
        exponential(const exponential& p);

        //- Construct and return a clone
        virtual autoPtr<distributionModel> clone() const
        {
            return autoPtr<distributionModel>(new exponential(*this));
        }

        //- No copy assignment
        void operator=(const exponential&) = delete;


    //- Destructor
    virtual ~exponential() = default;


    // Member Functions

        //- Sample the distribution
        virtual scalar sample() const;

        //- Return the theoretical mean of the distribution
        virtual scalar meanValue() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace distributionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
