/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::skewCorrected

Group
    grpFvSurfaceInterpolationSchemes

Description
    Skewness-corrected interpolation scheme that applies an explicit
    correction to given scheme.

SourceFiles
    skewCorrected.C

\*---------------------------------------------------------------------------*/

#ifndef gpuskewCorrected_H
#define gpuskewCorrected_H

#include "gpusurfaceInterpolationScheme.H"
#include "gpuskewCorrectionVectors.H"
#include "gpulinear.H"
#include "gpugaussGrad.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                       Class skewCorrected Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpuskewCorrected
:
    public gpusurfaceInterpolationScheme<Type>
{
    // Private member data

        tmp<gpusurfaceInterpolationScheme<Type>> tScheme_;


    // Private Member Functions

        //- No copy construct
        gpuskewCorrected(const gpuskewCorrected&) = delete;

        //- No copy assignment
        void operator=(const gpuskewCorrected&) = delete;


public:

    //- Runtime type information
    TypeName("skewCorrected");


    // Constructors

        //- Construct from mesh and Istream
        gpuskewCorrected
        (
            const gpufvMesh& mesh,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh),
            tScheme_
            (
                gpusurfaceInterpolationScheme<Type>::New(mesh, is)
            )
        {}


        //- Construct from mesh, faceFlux and Istream
        gpuskewCorrected
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh),
            tScheme_
            (
                gpusurfaceInterpolationScheme<Type>::New(mesh, faceFlux, is)
            )
        {}


    // Member Functions

        //- Return the interpolation weighting factors
        tmp<surfaceScalargpuField> weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            return tScheme_().weights(vf);
        }

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const
        {
            return
                tScheme_().corrected()
             || gpuskewCorrectionVectors::New(this->mesh()).skew();
        }

        tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        skewCorrection
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            const gpufvMesh& mesh = this->mesh();

            const gpuskewCorrectionVectors& scv = gpuskewCorrectionVectors::New(mesh);

            tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> tsfCorr
            (
                new GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>
                (
                    IOobject
                    (
                        "gpuskewCorrected::skewCorrection(" + vf.name() + ')',
                        mesh.time().timeName(),
                        mesh.hostmesh()
                    ),
                    mesh,
                    dimensioned<Type>(vf.dimensions(), Zero)
                )
            );

            for (direction cmpt=0; cmpt<pTraits<Type>::nComponents; cmpt++)
            {
                tsfCorr.ref().replace
                (
                    cmpt,
                    scv() & gpulinear
                    <
                        typename outerProduct
                        <
                            vector,
                            typename pTraits<Type>::cmptType
                        >::type
                    > (mesh).interpolate
                    (
                        fv::gpugaussGrad<typename pTraits<Type>::cmptType>
                        (mesh).grad(vf.component(cmpt))
                    )
                );
            }

            return tsfCorr;
        }


        //- Return the explicit correction to the face-interpolate
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            if
            (
                tScheme_().corrected()
             && gpuskewCorrectionVectors::New(this->mesh()).skew()
            )
            {
                return tScheme_().correction(vf) + skewCorrection(vf);
            }
            else if (tScheme_().corrected())
            {
                return tScheme_().correction(vf);
            }
            else if (gpuskewCorrectionVectors::New(this->mesh()).skew())
            {
                return skewCorrection(vf);
            }
            else
            {
                return
                    tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
                    (
                        nullptr
                    );
            }
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
