/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::CoBlended

Group
    grpFvSurfaceInterpolationSchemes

Description
    Two-scheme Courant number based blending differencing scheme.

    Similar to localBlended but uses a blending factor computed from the
    face-based Courant number and the lower and upper Courant number limits
    supplied:
    \f[
        weight = 1 - max(min((Co - Co1)/(Co2 - Co1), 1), 0)
    \f]
    where
    \vartable
        Co1 | Courant number below which scheme1 is used
        Co2 | Courant number above which scheme2 is used
    \endvartable

    The weight applies to the first scheme and 1-weight to the second scheme.

Usage
    Example of the CoBlended scheme specification using LUST for Courant numbers
    less than 1 and linearUpwind for Courant numbers greater than 10:
    \verbatim
    divSchemes
    {
        .
        .
        div(phi,U)      Gauss CoBlended 1 LUST grad(U) 10 linearUpwind grad(U);
        .
        .
    }
    \endverbatim

SourceFiles
    CoBlended.C

\*---------------------------------------------------------------------------*/

#ifndef gpuCoBlended_H
#define gpuCoBlended_H

#include "gpusurfaceInterpolationScheme.H"
#include "gpublendedSchemeBase.H"
#include "gpusurfaceInterpolate.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                        Class CoBlended Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpuCoBlended
:
    public gpusurfaceInterpolationScheme<Type>,
    public gpublendedSchemeBase<Type>
{
    // Private data

        //- Courant number below which scheme1 is used
        const scalar Co1_;

        //- Scheme 1
        tmp<gpusurfaceInterpolationScheme<Type>> tScheme1_;

        //- Courant number above which scheme2 is used
        const scalar Co2_;

        //- Scheme 2
        tmp<gpusurfaceInterpolationScheme<Type>> tScheme2_;

        //- The face-flux used to compute the face Courant number
        const surfaceScalargpuField& faceFlux_;


    // Private Member Functions

        //- No copy construct
        gpuCoBlended(const gpuCoBlended&) = delete;

        //- No copy assignment
        void operator=(const gpuCoBlended&) = delete;


public:

    //- Runtime type information
    TypeName("CoBlended");


    // Constructors

        //- Construct from mesh and Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpuCoBlended
        (
            const gpufvMesh& mesh,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh),
            Co1_(readScalar(is)),
            tScheme1_
            (
                gpusurfaceInterpolationScheme<Type>::New(mesh, is)
            ),
            Co2_(readScalar(is)),
            tScheme2_
            (
                gpusurfaceInterpolationScheme<Type>::New(mesh, is)
            ),
            faceFlux_
            (
                mesh.hostmesh().lookupObject<surfaceScalargpuField>(word(is))
            )
        {
            if (Co1_ < 0 || Co2_ < 0 || Co1_ >= Co2_)
            {
                FatalIOErrorInFunction(is)
                    << "coefficients = " << Co1_ << " and " << Co2_
                    << " should be > 0 and Co2 > Co1"
                    << exit(FatalIOError);
            }
        }


        //- Construct from mesh, faceFlux and Istream
        gpuCoBlended
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh),
            Co1_(readScalar(is)),
            tScheme1_
            (
                gpusurfaceInterpolationScheme<Type>::New(mesh, faceFlux, is)
            ),
            Co2_(readScalar(is)),
            tScheme2_
            (
                gpusurfaceInterpolationScheme<Type>::New(mesh, faceFlux, is)
            ),
            faceFlux_(faceFlux)
        {
            if (Co1_ < 0 || Co2_ < 0 || Co1_ >= Co2_)
            {
                FatalIOErrorInFunction(is)
                    << "coefficients = " << Co1_ << " and " << Co2_
                    << " should be > 0 and Co2 > Co1"
                    << exit(FatalIOError);
            }
        }


    // Member Functions

        //- Return the face-based blending factor
        virtual tmp<surfaceScalargpuField> blendingFactor
        (
             const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            const gpufvMesh& mesh = this->mesh();
            tmp<surfaceScalargpuField> tUflux = faceFlux_;

            if (faceFlux_.dimensions() == dimDensity*dimVelocity*dimArea)
            {
                // Currently assume that the density field
                // corresponding to the mass-flux is named "rho"
                const volScalargpuField& rho =
                    mesh.hostmesh().objectRegistry::template lookupObject<volScalargpuField>
                    ("rho");

                tUflux = faceFlux_/fvc::interpolate(rho);
            }
            else if (faceFlux_.dimensions() != dimVelocity*dimArea)
            {
                FatalErrorInFunction
                    << "dimensions of faceFlux are not correct"
                    << exit(FatalError);
            }

            return tmp<surfaceScalargpuField>
            (
                new surfaceScalargpuField
                (
                    vf.name() + "BlendingFactor",
                    scalar(1)
                  - max
                    (
                        min
                        (
                            (
                                mesh.time().deltaT()*mesh.deltaCoeffs()
                               *mag(tUflux)/mesh.magSf()
                              - Co1_
                            )/(Co2_ - Co1_),
                            scalar(1)
                        ),
                        scalar(0)
                    )
                )
            );
        }


        //- Return the interpolation weighting factors
        tmp<surfaceScalargpuField>
        weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            surfaceScalargpuField bf(blendingFactor(vf));

            return
                bf*tScheme1_().weights(vf)
              + (scalar(1) - bf)*tScheme2_().weights(vf);
        }


        //- Return the face-interpolate of the given cell field
        //  with explicit correction
        tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            surfaceScalargpuField bf(blendingFactor(vf));

            return
                bf*tScheme1_().interpolate(vf)
              + (scalar(1) - bf)*tScheme2_().interpolate(vf);
        }


        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const
        {
            return tScheme1_().corrected() || tScheme2_().corrected();
        }


        //- Return the explicit correction to the face-interpolate
        //  for the given field
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const
        {
            surfaceScalargpuField bf(blendingFactor(vf));

            if (tScheme1_().corrected())
            {
                if (tScheme2_().corrected())
                {
                    return
                    (
                        bf
                      * tScheme1_().correction(vf)
                      + (scalar(1) - bf)
                      * tScheme2_().correction(vf)
                    );
                }
                else
                {
                    return
                    (
                        bf
                      * tScheme1_().correction(vf)
                    );
                }
            }
            else if (tScheme2_().corrected())
            {
                return
                (
                    (scalar(1) - bf)
                  * tScheme2_().correction(vf)
                );
            }
            else
            {
                return tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
                (
                    nullptr
                );
            }
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
