/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::multivariateSurfaceInterpolationScheme

Description
    Abstract base class for multi-variate surface interpolation schemes.

SourceFiles
    multivariateSurfaceInterpolationScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpumultivariateSurfaceInterpolationScheme_H
#define gpumultivariateSurfaceInterpolationScheme_H

#include "gpusurfaceInterpolationScheme.H"
#include "HashTable.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
           Class multivariateSurfaceInterpolationScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpumultivariateSurfaceInterpolationScheme
:
    public refCount
{

public:

    //- fieldTable
    class fieldTable
    :
        public HashTable<const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>*>
    {
    public:

        fieldTable()
        {}

        void add(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& f)
        {
            this->insert(f.name(), &f);
        }
    };


private:

    // Private data

        //- Hold reference to mesh
        const gpufvMesh& mesh_;

        //- HashTable of pointers to the field set
        const fieldTable& fields_;


    // Private Member Functions

        //- No copy construct
        gpumultivariateSurfaceInterpolationScheme
        (
            const gpumultivariateSurfaceInterpolationScheme&
        ) = delete;

        //- No copy assignment
        void operator=(const gpumultivariateSurfaceInterpolationScheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpumultivariateSurfaceInterpolationScheme,
            Istream,
            (
                const gpufvMesh& mesh,
                const fieldTable& fields,
                const surfaceScalargpuField& faceFlux,
                Istream& is
            ),
            (mesh, fields, faceFlux, is)
        );


    // Constructors

        //- Construct for interpolating given field
        gpumultivariateSurfaceInterpolationScheme
        (
            const gpufvMesh& mesh,
            const fieldTable& fields,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


    // Selectors

        //- Return a pointer to a new gradScheme created on freestore
        static tmp<gpumultivariateSurfaceInterpolationScheme<Type>> New
        (
            const gpufvMesh& mesh,
            const fieldTable& fields,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpumultivariateSurfaceInterpolationScheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        //- Return fields to be interpolated
        const fieldTable& fields() const
        {
            return fields_;
        }


    // Member Operators

        //- surfaceInterpolationScheme sub-class returned by operator(field)
        //  which is used as the interpolation scheme for the field
        class fieldScheme
        :
            public gpusurfaceInterpolationScheme<Type>
        {

        public:

            // Constructors

                fieldScheme
                (
                    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
                )
                :
                    gpusurfaceInterpolationScheme<Type>(field.mesh())
                {}


            // Member Functions

                //- Return the interpolation weighting factors
                virtual tmp<surfaceScalargpuField> weights
                (
                    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
                ) const = 0;
        };

        virtual tmp<gpusurfaceInterpolationScheme<Type>> operator()
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
        ) const = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeMultivariateSurfaceInterpolationTypeScheme(SS, Type)               \
                                                                               \
defineNamedTemplateTypeNameAndDebug(SS<Type>, 0);                              \
                                                                               \
gpumultivariateSurfaceInterpolationScheme<Type>::                                 \
addIstreamConstructorToTable<SS<Type>>                                         \
    add##SS##Type##ConstructorToTable_;


#define makeMultivariateSurfaceInterpolationScheme(SS)                         \
                                                                               \
makeMultivariateSurfaceInterpolationTypeScheme(SS, scalar)                     \
makeMultivariateSurfaceInterpolationTypeScheme(SS, vector)                     \
makeMultivariateSurfaceInterpolationTypeScheme(SS, sphericalTensor)            \
makeMultivariateSurfaceInterpolationTypeScheme(SS, symmTensor)                 \
makeMultivariateSurfaceInterpolationTypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpumultivariateSurfaceInterpolationScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
