/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::limitedSurfaceInterpolationScheme

Group
    grpFvLimitedSurfaceInterpolationSchemes

Description
    Abstract base class for limited surface interpolation schemes.

SourceFiles
    limitedSurfaceInterpolationScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpulimitedSurfaceInterpolationScheme_H
#define gpulimitedSurfaceInterpolationScheme_H

#include "gpusurfaceInterpolationScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                 Class limitedSurfaceInterpolationScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulimitedSurfaceInterpolationScheme
:
    public gpusurfaceInterpolationScheme<Type>
{
    // Private Member Functions

        //- No copy construct
        gpulimitedSurfaceInterpolationScheme
        (
            const gpulimitedSurfaceInterpolationScheme&
        ) = delete;

        //- No copy assignment
        void operator=(const gpulimitedSurfaceInterpolationScheme&) = delete;


protected:

    // Protected data

        const surfaceScalargpuField& faceFlux_;


public:

    //- Runtime type information
    TypeName("limitedSurfaceInterpolationScheme");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpulimitedSurfaceInterpolationScheme,
            Mesh,
            (
                const gpufvMesh& mesh,
                Istream& schemeData
            ),
            (mesh, schemeData)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            gpulimitedSurfaceInterpolationScheme,
            MeshFlux,
            (
                const gpufvMesh& mesh,
                const surfaceScalargpuField& faceFlux,
                Istream& schemeData
            ),
            (mesh, faceFlux, schemeData)
        );


    // Constructors

        //- Construct from mesh and faceFlux
        gpulimitedSurfaceInterpolationScheme
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh),
            faceFlux_(faceFlux)
        {}


        //- Construct from mesh and Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpulimitedSurfaceInterpolationScheme
        (
            const gpufvMesh& mesh,
            Istream& is
        )
        :
            gpusurfaceInterpolationScheme<Type>(mesh),
            faceFlux_
            (
                mesh.hostmesh().lookupObject<surfaceScalargpuField>
                (
                    word(is)
                )
            )
        {}


    // Selectors

        //- Return new tmp interpolation scheme
        static tmp<gpulimitedSurfaceInterpolationScheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );

        //- Return new tmp interpolation scheme
        static tmp<gpulimitedSurfaceInterpolationScheme<Type>> New
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpulimitedSurfaceInterpolationScheme();


    // Member Functions

        //- Return the interpolation weighting factors
        virtual tmp<surfaceScalargpuField> limiter
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;

        //- Return the interpolation weighting factors for the given field,
        //  by limiting the given weights with the given limiter
        tmp<surfaceScalargpuField> weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const surfaceScalargpuField& CDweights,
            tmp<surfaceScalargpuField> tLimiter
        ) const;

        //- Return the interpolation weighting factors for the given field
        virtual tmp<surfaceScalargpuField> weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;

        //- Return the interpolation weighting factors
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        flux
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makegpulimitedSurfaceInterpolationTypeScheme(SS, Type)                    \
                                                                               \
defineNamedTemplateTypeNameAndDebug(SS<Type>, 0);                              \
                                                                               \
gpusurfaceInterpolationScheme<Type>::addMeshConstructorToTable<SS<Type>>          \
    add##SS##Type##MeshConstructorToTable_;                                    \
                                                                               \
gpusurfaceInterpolationScheme<Type>::addMeshFluxConstructorToTable<SS<Type>>      \
    add##SS##Type##MeshFluxConstructorToTable_;                                \
                                                                               \
gpulimitedSurfaceInterpolationScheme<Type>::addMeshConstructorToTable<SS<Type>>   \
    add##SS##Type##MeshConstructorToLimitedTable_;                             \
                                                                               \
gpulimitedSurfaceInterpolationScheme<Type>::                                      \
    addMeshFluxConstructorToTable<SS<Type>>                                    \
    add##SS##Type##MeshFluxConstructorToLimitedTable_;

#define makegpulimitedSurfaceInterpolationScheme(SS)                              \
                                                                               \
makegpulimitedSurfaceInterpolationTypeScheme(SS, scalar)                          \
makegpulimitedSurfaceInterpolationTypeScheme(SS, vector)                          \
makegpulimitedSurfaceInterpolationTypeScheme(SS, sphericalTensor)                 \
makegpulimitedSurfaceInterpolationTypeScheme(SS, symmTensor)                      \
makegpulimitedSurfaceInterpolationTypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpulimitedSurfaceInterpolationScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
