/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::PhiScheme

Group
    grpFvLimitedSurfaceInterpolationSchemes

Description
    Class to create the weighting-factors based on the face-flux.

    The particular differencing scheme class is supplied as a template
    argument, the weight function of which is called by the weight function
    of this class for the internal faces as well as faces of coupled
    patches (e.g. processor-processor patches). The weight function is
    supplied with the central-differencing weighting factor, the face-flux,
    the face neighbour cell values and the face area.

    This code organisation is both neat and efficient, allowing for
    convenient implementation of new schemes to run on parallelised cases.

SourceFiles
    PhiScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpuPhiScheme_H
#define gpuPhiScheme_H

#include "gpulimitedSurfaceInterpolationScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class PhiScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type, class gpuPhiLimiter>
class gpuPhiScheme
:
    public gpulimitedSurfaceInterpolationScheme<Type>,
    public gpuPhiLimiter
{
    // Private Member Functions

        //- No copy construct
        gpuPhiScheme(const gpuPhiScheme&) = delete;

        //- No copy assignment
        void operator=(const gpuPhiScheme&) = delete;


public:

    //- Runtime type information
    TypeName("gpuPhiScheme");


    // Constructors

        //- Construct from mesh, faceFlux and blendingFactor
        gpuPhiScheme
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            const gpuPhiLimiter& weight
        )
        :
            gpulimitedSurfaceInterpolationScheme<Type>(mesh, faceFlux),
            gpuPhiLimiter(weight)
        {}

        //- Construct from mesh and Istream.
        //  The name of the flux field is read from the Istream and looked-up
        //  from the mesh objectRegistry
        gpuPhiScheme
        (
            const gpufvMesh& mesh,
            Istream& is
        )
        :
            gpulimitedSurfaceInterpolationScheme<Type>(mesh, is),
            gpuPhiLimiter(is)
        {}

        //- Construct from mesh, faceFlux and Istream
        gpuPhiScheme
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& is
        )
        :
            gpulimitedSurfaceInterpolationScheme<Type>(mesh, faceFlux),
            gpuPhiLimiter(is)
        {}


    // Member Functions

        //- Return the interpolation weighting factors
        virtual tmp<surfaceScalargpuField> limiter
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makegpuPhiSurfaceInterpolationScheme(SS, WEIGHT, TYPE)                    \
                                                                               \
typedef gpuPhiScheme<TYPE, WEIGHT> gpuPhischeme##WEIGHT_;                            \
defineTemplateTypeNameAndDebugWithName(gpuPhischeme##WEIGHT_, #SS, 0);            \
                                                                               \
gpusurfaceInterpolationScheme<TYPE>::addMeshConstructorToTable                    \
<gpuPhiScheme<TYPE, WEIGHT>> add##SS##TYPE##MeshConstructorToTable_;              \
                                                                               \
gpusurfaceInterpolationScheme<TYPE>::addMeshFluxConstructorToTable                \
<gpuPhiScheme<TYPE, WEIGHT>> add##SS##TYPE##MeshFluxConstructorToTable_;          \
                                                                               \
gpulimitedSurfaceInterpolationScheme<TYPE>::addMeshConstructorToTable             \
<gpuPhiScheme<TYPE, WEIGHT>> add##SS##TYPE##MeshConstructorToLimitedTable_;       \
                                                                               \
gpulimitedSurfaceInterpolationScheme<TYPE>::addMeshFluxConstructorToTable         \
<gpuPhiScheme<TYPE, WEIGHT>> add##SS##TYPE##MeshFluxConstructorToLimitedTable_;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuPhiScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
