/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2018 OpenFOAM Foundation
    Copyright (C) 2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvPatch

Description
    A finiteVolume patch using a polyPatch and a fvBoundaryMesh

SourceFiles
    fvPatch.C
    fvPatchNew.C

\*---------------------------------------------------------------------------*/

#ifndef gpufvPatch_H
#define gpufvPatch_H

#include "polyPatch.H"
#include "labelList.H"
#include "SubList.H"
#include "typeInfo.H"
#include "tmp.H"
#include "primitiveFields.H"
#include "SubField.H"
#include "fvPatchgpuFieldsFwd.H"
#include "autoPtr.H"
#include "runTimeSelectionTables.H"
#include "fvPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward Declarations
class gpufvBoundaryMesh;
class gpusurfaceInterpolation;

/*---------------------------------------------------------------------------*\
                           Class fvPatch Declaration
\*---------------------------------------------------------------------------*/

class gpufvPatch
{
    // Private Data

        //- Reference to the underlying polyPatch
        const polyPatch& polyPatch_;

        //- Reference to boundary mesh
        const gpufvBoundaryMesh& boundaryMesh_;


    // Private Member Functions

        //- No copy construct
        gpufvPatch(const gpufvPatch&) = delete;

        //- No copy assignment
        void operator=(const gpufvPatch&) = delete;


public:

    // Protected Member Functions

        //- Make patch weighting factors
        virtual void makeWeights(scalargpuField&) const;

        //- Correct patch deltaCoeffs
        virtual void makeDeltaCoeffs(scalargpuField&) const;

        //- Correct patch non-ortho deltaCoeffs
        virtual void makeNonOrthoDeltaCoeffs(scalargpuField&) const;

        //- Correct patch non-ortho correction vectors
        virtual void makeNonOrthoCorrVectors(vectorgpuField&) const;

        //- Initialise the patches for moving points
        virtual void initMovePoints();

        //- Correct patches after moving points
        virtual void movePoints();


public:

    typedef gpufvBoundaryMesh BoundaryMesh;

    friend class gpufvBoundaryMesh;
    friend class gpusurfaceInterpolation;

    //- Runtime type information
    TypeName(polyPatch::typeName_());


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            autoPtr,
            gpufvPatch,
            polyPatch,
            (const polyPatch& patch, const gpufvBoundaryMesh& bm),
            (patch, bm)
        );


    // Constructors

        //- Construct from polyPatch and fvBoundaryMesh
        gpufvPatch(const polyPatch&, const gpufvBoundaryMesh&);


    // Selectors

        //- Return a pointer to a new patch created on freestore from polyPatch
        static autoPtr<gpufvPatch> New
        (
            const polyPatch&,
            const gpufvBoundaryMesh&
        );


    //- Destructor
    virtual ~gpufvPatch();


    // Member Functions

        //- Lookup the polyPatch index on corresponding fvMesh
        //  \note Fatal if the polyPatch is not associated with a fvMesh
        static const fvPatch& lookupPatch(const polyPatch& p);


    // Access

            //- Return the polyPatch
            const polyPatch& patch() const
            {
                return polyPatch_;
            }

            //- Return name
            virtual const word& name() const
            {
                return polyPatch_.name();
            }

            //- Return start label of this patch in the polyMesh face list
            virtual label start() const
            {
                return polyPatch_.start();
            }

            //- Return size
            virtual label size() const
            {
                return polyPatch_.size();
            }

            //- Return true if this patch is coupled
            virtual bool coupled() const
            {
                return polyPatch_.coupled();
            }

            //- Return true if the given type is a constraint type
            static bool constraintType(const word& pt);

            //- Return a list of all the constraint patch types
            static wordList constraintTypes();

            //- Return the index of this patch in the fvBoundaryMesh
            label index() const
            {
                return polyPatch_.index();
            }

            //- Return boundaryMesh reference
            const gpufvBoundaryMesh& boundaryMesh() const
            {
                return boundaryMesh_;
            }

            //- Slice list to patch
            template<class T>
            const typename List<T>::subList patchSlice(const List<T>& l) const
            {
                return typename List<T>::subList(l, size(), start());
            }

            //- Return faceCells
            virtual const labelgpuList& gpuFaceCells() const;

        // Access functions for geometrical data

            //- Return face centres
            const vectorgpuField& Cf() const;

            //- Return neighbour cell centres
            tmp<vectorgpuField> Cn() const;

            //- Return face area vectors
            const vectorgpuField& Sf() const;

            //- Return face area magnitudes
            const scalargpuField& magSf() const;

            //- Return face normals
            tmp<vectorgpuField> nf() const;

            //- Return cell-centre to face-centre vector
            //- except for coupled patches for which the cell-centre
            //- to coupled-cell-centre vector is returned
            virtual tmp<vectorgpuField> delta() const;


        // Access functions for demand driven data

            //- Return patch weighting factors
            const scalargpuField& weights() const;

            //- Return the face - cell distance coefficient
            //- except for coupled patches for which the cell-centre
            //- to coupled-cell-centre distance coefficient is returned
            const scalargpuField& deltaCoeffs() const;


        // Evaluation functions

            //- Return given internal field next to patch as patch field
            template<class Type>
            tmp<gpuField<Type>> patchInternalField(const gpuList<Type>&) const;

            //- Return given internal field next to patch as patch field
            //- using faceCells mapping
            template<class Type>
            tmp<gpuField<Type>> patchInternalField
            (
                const gpuList<Type>&,
                const labelgpuList& faceCells
            ) const;

            //- Return given internal field next to patch as patch field
            template<class Type>
            void patchInternalField(const gpuList<Type>&, gpuField<Type>&) const;

            //- Return the corresponding patchField of the named field
            template<class GeometricgpuField, class Type>
            const typename GeometricgpuField::Patch& patchField
            (
                const GeometricgpuField&
            ) const;

            //- Lookup and return the patchField of the named field from the
            //- local objectRegistry.
            //  N.B.  The dummy pointer arguments are used if this function is
            //  instantiated within a templated function to avoid a bug in gcc.
            //  See inletOutletFvPatchField.C and outletInletFvPatchField.C
            template<class GeometricgpuField, class Type>
            const typename GeometricgpuField::Patch& lookupPatchField
            (
                const word& name,
                const GeometricgpuField* = nullptr,
                const Type* = nullptr
            ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpufvPatchTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
