/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::cyclicFvPatch

Description
    Cyclic-plane patch.

SourceFiles
    cyclicFvPatch.C

\*---------------------------------------------------------------------------*/

#ifndef cyclicgpuFvPatch_H
#define cyclicgpuFvPatch_H

#include "coupledgpuFvPatch.H"
#include "cyclicgpuLduInterface.H"
#include "cyclicPolyPatch.H"
#include "gpufvBoundaryMesh.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class cyclicFvPatch Declaration
\*---------------------------------------------------------------------------*/

class cyclicgpuFvPatch
:
    public coupledgpuFvPatch,
    public cyclicgpuLduInterface
{
    // Private data

        const cyclicPolyPatch& cyclicPolyPatch_;


protected:

    // Protected Member functions

        //- Make patch weighting factors
        void makeWeights(scalargpuField&) const;


public:

    //- Runtime type information
    TypeName(cyclicPolyPatch::typeName_());


    // Constructors

        //- Construct from polyPatch
        cyclicgpuFvPatch(const polyPatch& patch, const gpufvBoundaryMesh& bm)
        :
            coupledgpuFvPatch(patch, bm),
            cyclicPolyPatch_(refCast<const cyclicPolyPatch>(patch))
        {}


    // Member functions


        // Access

            //- Return local reference cast into the cyclic patch
            const cyclicPolyPatch& cyclicPatch() const
            {
                return cyclicPolyPatch_;
            }

            //- Return neighbour
            virtual label neighbPatchID() const
            {
                return cyclicPolyPatch_.neighbPatchID();
            }

            virtual bool owner() const
            {
                return cyclicPolyPatch_.owner();
            }

            //- Return processor number
            virtual const cyclicgpuFvPatch& neighbPatch() const
            {
                return refCast<const cyclicgpuFvPatch>
                (
                    this->boundaryMesh()[cyclicPolyPatch_.neighbPatchID()]
                );
            }

            //- Are the cyclic planes parallel
            virtual bool parallel() const
            {
                return cyclicPolyPatch_.parallel();
            }

            //- Return face transformation tensor

            virtual const tensorgpuField& gpuForwardT() const
            {
                return cyclicPolyPatch_.gpuForwardT();
            }

            //- Return neighbour-cell transformation tensor

            virtual const tensorgpuField& gpuReverseT() const
            {
                return cyclicPolyPatch_.gpuReverseT();
            }

            const cyclicgpuFvPatch& neighbFvPatch() const
            {
                return refCast<const cyclicgpuFvPatch>
                (
                    this->boundaryMesh()[cyclicPolyPatch_.neighbPatchID()]
                );
            }


            //- Return delta (P to N) vectors across coupled patch
            virtual tmp<vectorgpuField> delta() const;


        // Interface transfer functions

            //- Return the values of the given internal data adjacent to
            //- the interface as a field
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData
            ) const;


            //- Return the values of the given internal data adjacent to
            //- the interface as a field using a mapping faceCell
            virtual tmp<labelgpuField> interfaceInternalField
            (
                const labelgpuList& internalData,
                const labelgpuList& faceCell
            ) const;


            //- Return neighbour field
            virtual tmp<labelgpuField> internalFieldTransfer
            (
                const Pstream::commsTypes commsType,
                const labelgpuList& internalData
            ) const;

};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
