/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2020-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvMesh.H"
#include "Time.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"
#include "slicedVolgpuFields.H"
#include "slicedSurfacegpuFields.H"
#include "cyclicFvPatchgpuFields.H"
#include "cyclicAMIFvPatchgpuFields.H"

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

void Foam::gpufvMesh::makeSf() const
{
    DebugInFunction << "Assembling face areas" << endl;

    // It is an error to attempt to recalculate
    // if the pointer is already set
    if (SfPtr_)
    {
        FatalErrorInFunction
            << "face areas already exist"
            << abort(FatalError);
    }

    SfPtr_ = new slicedSurfaceVectorgpuField
    (
        IOobject
        (
            "S",
            hostmesh().pointsInstance(),
            hostmesh().meshSubDir,
            hostmesh_,
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false
        ),
        *this,
        dimArea,
        hostmesh().gpuFaceAreas()
    );

    SfPtr_->setOriented();
}


void Foam::gpufvMesh::makeMagSf() const
{
    DebugInFunction << "Assembling mag face areas" << endl;

    // It is an error to attempt to recalculate
    // if the pointer is already set
    if (magSfPtr_)
    {
        FatalErrorInFunction
            << "mag face areas already exist"
            << abort(FatalError);
    }

    // Note: Added stabilisation for faces with exactly zero area.
    // These should be caught on mesh checking but at least this stops
    // the code from producing Nans.
    magSfPtr_ = new surfaceScalargpuField
    (
        IOobject
        (
            "magSf",
            hostmesh().pointsInstance(),
            hostmesh().meshSubDir,
            hostmesh_,
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false
        ),
        mag(Sf()) + dimensionedScalar("vs", dimArea, VSMALL)
    );
}


void Foam::gpufvMesh::makeC() const
{
    DebugInFunction << "Assembling cell centres" << endl;

    // It is an error to attempt to recalculate
    // if the pointer is already set
    if (CPtr_)
    {
        FatalErrorInFunction
            << "cell centres already exist"
            << abort(FatalError);
    }

    // Construct as slices. Only preserve processor (not e.g. cyclic)

    CPtr_ = new slicedVolVectorgpuField
    (
        IOobject
        (
            "C",
            hostmesh().pointsInstance(),
            hostmesh().meshSubDir,
            hostmesh_,
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false
        ),
        *this,
        dimLength,
        hostmesh().gpuCellCentres(),
        hostmesh().gpuFaceCentres(),
        true,               //preserveCouples
        true                //preserveProcOnly
    );
}


void Foam::gpufvMesh::makeCf() const
{
    DebugInFunction << "Assembling face centres" << endl;

    // It is an error to attempt to recalculate
    // if the pointer is already set
    if (CfPtr_)
    {
        FatalErrorInFunction
            << "face centres already exist"
            << abort(FatalError);
    }

    CfPtr_ = new slicedSurfaceVectorgpuField
    (
        IOobject
        (
            "Cf",
            hostmesh().pointsInstance(),
            hostmesh().meshSubDir,
            hostmesh_,
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false
        ),
        *this,
        dimLength,
        hostmesh().gpuFaceCentres()
    );
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

const Foam::volScalargpuField::Internal& Foam::gpufvMesh::V() const
{
    if (!VPtr_)
    {
        DebugInFunction
            << "Constructing from primitiveMesh::cellVolumes()" << endl;

        VPtr_ = new slicedVolScalargpuField::Internal
        (
            IOobject
            (
                "V",
                time().timeName(),
                hostmesh_,
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            *this,
            dimVolume,
            hostmesh().gpuCellVolumes()
        );
    }

    return *static_cast<slicedVolScalargpuField::Internal*>(VPtr_);
}


const Foam::volScalargpuField::Internal& Foam::gpufvMesh::V0() const
{
    if (!V0Ptr_)
    {
        FatalErrorInFunction
            << "V0 is not available"
            << abort(FatalError);
    }

    return *V0Ptr_;
}


Foam::volScalargpuField::Internal& Foam::gpufvMesh::setV0()
{
    if (!V0Ptr_)
    {
        FatalErrorInFunction
            << "V0 is not available"
            << abort(FatalError);
    }

    return *V0Ptr_;
}


const Foam::volScalargpuField::Internal& Foam::gpufvMesh::V00() const
{
    if (!V00Ptr_)
    {
        DebugInFunction << "Constructing from V0" << endl;

        V00Ptr_ = new DimensionedgpuField<scalar, gpuvolMesh>
        (
            V0()
        );
    }

    return *V00Ptr_;
}


Foam::tmp<Foam::volScalargpuField::Internal> Foam::gpufvMesh::Vsc() const
{
    if (!hostmesh().steady() && hostmesh().moving() && time().subCycling())
    {
        const TimeState& ts = time();
        const TimeState& ts0 = time().prevTimeState();

        scalar tFrac =
        (
            ts.value() - (ts0.value() - ts0.deltaTValue())
        )/ts0.deltaTValue();

        if (tFrac < (1 - SMALL))
        {
            return V0() + tFrac*(V() - V0());
        }
        else
        {
            return V();
        }
    }
    else
    {
        return V();
    }
}


Foam::tmp<Foam::volScalargpuField::Internal> Foam::gpufvMesh::Vsc0() const
{
    if (!hostmesh().steady() && hostmesh().moving() && time().subCycling())
    {
        const TimeState& ts = time();
        const TimeState& ts0 = time().prevTimeState();

        scalar t0Frac =
        (
            (ts.value() - ts.deltaTValue())
          - (ts0.value() - ts0.deltaTValue())
        )/ts0.deltaTValue();

        if (t0Frac > SMALL)
        {
            return V0() + t0Frac*(V() - V0());
        }
        else
        {
            return V0();
        }
    }
    else
    {
        return V0();
    }
}


const Foam::surfaceVectorgpuField& Foam::gpufvMesh::Sf() const
{
    if (!SfPtr_)
    {
        makeSf();
    }

    return *SfPtr_;
}


const Foam::surfaceScalargpuField& Foam::gpufvMesh::magSf() const
{
    if (!magSfPtr_)
    {
        makeMagSf();
    }

    return *magSfPtr_;
}


const Foam::volVectorgpuField& Foam::gpufvMesh::C() const
{
    if (!CPtr_)
    {
        makeC();
    }

    return *CPtr_;
}


const Foam::surfaceVectorgpuField& Foam::gpufvMesh::Cf() const
{
    if (!CfPtr_)
    {
        makeCf();
    }

    return *CfPtr_;
}

namespace Foam
{
struct calcFaceDeltaFunctor : public thrust::unary_function<thrust::tuple<vector,vector>,vector>
{
    __host__ __device__
    vector operator()(const thrust::tuple<vector,vector>& c)
    {
        return thrust::get<0>(c) - thrust::get<1>(c);
    }
};
}


Foam::tmp<Foam::surfaceVectorgpuField> Foam::gpufvMesh::delta() const
{
    DebugInFunction << "Calculating face deltas" << endl;

    tmp<surfaceVectorgpuField> tdelta
    (
        new surfaceVectorgpuField
        (
            IOobject
            (
                "delta",
                hostmesh().pointsInstance(),
                hostmesh().meshSubDir,
                hostmesh_,
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            *this,
            dimLength
        )
    );
    surfaceVectorgpuField& delta = tdelta.ref();
    delta.setOriented();

    const volVectorgpuField& C = this->C();
    const labelgpuList& owner = this->owner();
    const labelgpuList& neighbour = this->neighbour();

    thrust::copy
    (
        thrust::make_transform_iterator
        ( 
            thrust::make_zip_iterator(thrust::make_tuple
            ( 
                thrust::make_permutation_iterator( C.primitiveField().begin(),neighbour.begin() ),
                thrust::make_permutation_iterator( C.primitiveField().begin(),owner.begin() )
            )),
            calcFaceDeltaFunctor()
        ),
        thrust::make_transform_iterator
        (
            thrust::make_zip_iterator(thrust::make_tuple
            ( 
                thrust::make_permutation_iterator( C.primitiveField().begin(),neighbour.begin()+owner.size() ),
                thrust::make_permutation_iterator( C.primitiveField().begin(),owner.end() )
            )),
            calcFaceDeltaFunctor()
        ),
        delta.primitiveFieldRef().begin()
    );

    surfaceVectorgpuField::Boundary& deltabf =  delta.boundaryFieldRef();

    forAll(deltabf, patchi)
    {
        deltabf[patchi] = boundary()[patchi].delta();
    }

    return tdelta;
}


const Foam::surfaceScalargpuField& Foam::gpufvMesh::phi() const
{
    if (!phiPtr_)
    {
        FatalErrorInFunction
            << "mesh flux field does not exist, is the mesh actually moving?"
            << abort(FatalError);
    }

    // Set zero current time
    // mesh motion fluxes if the time has been incremented
    if (!time().subCycling() && phiPtr_->timeIndex() != time().timeIndex())
    {
        (*phiPtr_) = dimensionedScalar(dimVolume/dimTime, Zero);
    }

    phiPtr_->setOriented();

    return *phiPtr_;
}


Foam::surfaceScalargpuField& Foam::gpufvMesh::setPhi()
{
    if (!phiPtr_)
    {
        FatalErrorInFunction
            << "mesh flux field does not exist, is the mesh actually moving?"
            << abort(FatalError);
    }

    return *phiPtr_;
}


// ************************************************************************* //
