/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvsPatchgpuField

Description
    An abstract base class with a fat-interface to all derived classes
    covering all possible ways in which they might be used.

    The first level of derivation is to basic patchFields which cover
    zero-gradient, fixed-gradient, fixed-value and mixed conditions.

    The next level of derivation covers all the specialised typed with
    specific evaluation procedures, particularly with respect to specific
    fields.

SourceFiles
    fvsPatchgpuField.C
    fvsPatchgpuFieldNew.C

\*---------------------------------------------------------------------------*/

#ifndef fvsPatchgpuField_H
#define fvsPatchgpuField_H

#include "gpufvPatch.H"
#include "DimensionedgpuField.H"
#include "fvsPatchField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declaration of classes

class objectRegistry;
class dictionary;
class fvPatchgpuFieldMapper;
class gpusurfaceMesh;


// Forward declaration of friend functions and operators

template<class Type>
class fvsPatchgpuField;

template<class Type>
class calculatedFvsPatchgpuField;

template<class Type>
Ostream& operator<<(Ostream&, const fvsPatchgpuField<Type>&);


/*---------------------------------------------------------------------------*\
                           Class patch Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class fvsPatchgpuField
:
    public gpuField<Type>
{
    // Private data

        //- Reference to patch
        const gpufvPatch& patch_;

        //- Reference to internal field
        const DimensionedgpuField<Type, gpusurfaceMesh>& internalField_;


public:

    typedef gpufvPatch Patch;
    typedef calculatedFvsPatchgpuField<Type> Calculated;
//	typedef fvsPatchField<Type> hostType;


    //- Runtime type information
    TypeName("fvsPatchgpuField");

    //- Debug switch to disallow the use of genericFvsPatchField
    static int disallowGenericFvsPatchField;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            fvsPatchgpuField,
            patch,
            (
                const gpufvPatch& p,
                const DimensionedgpuField<Type, gpusurfaceMesh>& iF
            ),
            (p, iF)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            fvsPatchgpuField,
            patchMapper,
            (
                const fvsPatchgpuField<Type>& ptf,
                const gpufvPatch& p,
                const DimensionedgpuField<Type, gpusurfaceMesh>& iF,
                const fvPatchgpuFieldMapper& m
            ),
            (dynamic_cast<const fvsPatchgpuFieldType&>(ptf), p, iF, m)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            fvsPatchgpuField,
            dictionary,
            (
                const gpufvPatch& p,
                const DimensionedgpuField<Type, gpusurfaceMesh>& iF,
                const dictionary& dict
            ),
            (p, iF, dict)
        );


    // Constructors

        //- Construct from patch and internal field
        fvsPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&
        );

        //- Construct from patch and internal field and patch field
        fvsPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&,
            const gpuField<Type>&
        );

        //- Construct from patch, internal field and dictionary
        fvsPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&,
            const dictionary&
        );

        //- Construct by mapping the given fvsPatchgpuField onto a new patch
        fvsPatchgpuField
        (
            const fvsPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        fvsPatchgpuField(const fvsPatchgpuField<Type>&);

        //- Construct and return a clone
        virtual tmp<fvsPatchgpuField<Type>> clone() const
        {
            return tmp<fvsPatchgpuField<Type>>::New(*this);
        }

        //- Construct as copy setting internal field reference
        fvsPatchgpuField
        (
            const fvsPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvsPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpusurfaceMesh>& iF
        ) const
        {
            return tmp<fvsPatchgpuField<Type>>::New(*this, iF);
        }


    // Selectors

        //- Return a pointer to a new patchField created on freestore given
        //  patch and internal field
        //  (does not set the patch field values)
        static tmp<fvsPatchgpuField<Type>> New
        (
            const word&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&
        );

        //- Return a pointer to a new patchField created on freestore given
        //  patch and internal field
        //  (does not set the patch field values)
        //  Allows override of constraint type
        static tmp<fvsPatchgpuField<Type>> New
        (
            const word&,
            const word& actualPatchType,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&
        );

        //- Return a pointer to a new patchField created on freestore from
        //  a given fvsPatchgpuField mapped onto a new patch
        static tmp<fvsPatchgpuField<Type>> New
        (
            const fvsPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Return a pointer to a new patchField created on freestore
        //  from dictionary
        static tmp<fvsPatchgpuField<Type>> New
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpusurfaceMesh>&,
            const dictionary&
        );

        //- Return a pointer to a new calculatedFvsPatchgpuField created on
        //  freestore without setting patchField values
        static tmp<fvsPatchgpuField<Type>> NewCalculatedType
        (
            const gpufvPatch&
        );

        //- Return a pointer to a new calculatedFvsPatchgpuField created on
        //  freestore without setting patchField values
        template<class Type2>
        static tmp<fvsPatchgpuField<Type>> NewCalculatedType
        (
            const fvsPatchgpuField<Type2>&
        );


    //- Destructor
    virtual ~fvsPatchgpuField<Type>() = default;


    // Member functions

        // Access

            //- Return local objectRegistry
            const objectRegistry& db() const;

            //- Return patch
            const gpufvPatch& patch() const
            {
                return patch_;
            }

            //- Return dimensioned internal field reference
            const DimensionedgpuField<Type, gpusurfaceMesh>& internalField() const
            {
                return internalField_;
            }

            //- Return internal field reference
            const gpuField<Type>& primitiveField() const
            {
                return internalField_;
            }

            //- Return the type of the calculated for of fvsPatchgpuField
            static const word& calculatedType();

            //- Return true if this patch field fixes a value.
            //  Needed to check if a level has to be specified while solving
            //  Poissons equations.
            virtual bool fixesValue() const
            {
                return false;
            }

            //- Return true if this patch field is coupled
            virtual bool coupled() const
            {
                return false;
            }


        // Mapping functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap
            (
                const fvPatchgpuFieldMapper&
            );

            //- Reverse map the given fvsPatchgpuField onto this fvsPatchgpuField
            virtual void rmap
            (
                const fvsPatchgpuField<Type>&,
                const labelgpuList&
            );


        //- Write
        virtual void write(Ostream&) const;


        // Check

            //- Check fvsPatchgpuField<Type> against given fvsPatchgpuField<Type>
            void check(const fvsPatchgpuField<Type>&) const;


    // Member operators

        virtual void operator=(const UList<Type>&);
        virtual void operator=(const gpuList<Type>&);

        virtual void operator=(const fvsPatchgpuField<Type>&);
        virtual void operator+=(const fvsPatchgpuField<Type>&);
        virtual void operator-=(const fvsPatchgpuField<Type>&);
        virtual void operator*=(const fvsPatchgpuField<scalar>&);
        virtual void operator/=(const fvsPatchgpuField<scalar>&);

        virtual void operator+=(const gpuField<Type>&);
        virtual void operator-=(const gpuField<Type>&);

        virtual void operator*=(const gpuField<scalar>&);
        virtual void operator/=(const gpuField<scalar>&);

        virtual void operator=(const Type&);
        virtual void operator+=(const Type&);
        virtual void operator-=(const Type&);
        virtual void operator*=(const scalar);
        virtual void operator/=(const scalar);


        // Force an assignment irrespective of form of patch

        virtual void operator==(const fvsPatchgpuField<Type>&);
        virtual void operator==(const gpuField<Type>&);
        virtual void operator==(const Type&);


    // Ostream operator

        friend Ostream& operator<< <Type>(Ostream&, const fvsPatchgpuField<Type>&);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "fvsPatchgpuField.C"
    #include "calculatedFvsPatchgpuField.H"
#endif


#define makeFvsPatchTypeFieldTypeName(type)                                    \
                                                                               \
defineNamedTemplateTypeNameAndDebug(type, 0);

#define makeFvsPatchFieldsTypeName(type)                                       \
                                                                               \
makeFvsPatchTypeFieldTypeName(type##FvsPatchScalargpuField);                      \
makeFvsPatchTypeFieldTypeName(type##FvsPatchVectorgpuField);                      \
makeFvsPatchTypeFieldTypeName(type##FvsPatchSphericalTensorgpuField);             \
makeFvsPatchTypeFieldTypeName(type##FvsPatchSymmTensorgpuField);                  \
makeFvsPatchTypeFieldTypeName(type##FvsPatchTensorgpuField);

#define makeFvsPatchTypeField(PatchTypeField, typePatchTypeField)              \
                                                                               \
defineNamedTemplateTypeNameAndDebug(typePatchTypeField, 0);                    \
                                                                               \
addToRunTimeSelectionTable                                                     \
(                                                                              \
    PatchTypeField, typePatchTypeField, patch                                  \
);                                                                             \
                                                                               \
addToRunTimeSelectionTable                                                     \
(                                                                              \
    PatchTypeField,                                                            \
    typePatchTypeField,                                                        \
    patchMapper                                                                \
);                                                                             \
                                                                               \
addToRunTimeSelectionTable                                                     \
(                                                                              \
    PatchTypeField, typePatchTypeField, dictionary                             \
);


#define makeFvsPatchFields(type)                                               \
                                                                               \
makeFvsPatchTypeField(fvsPatchScalargpuField, type##FvsPatchScalargpuField);         \
makeFvsPatchTypeField(fvsPatchVectorgpuField, type##FvsPatchVectorgpuField);         \
makeFvsPatchTypeField                                                          \
(                                                                              \
    fvsPatchSphericalTensorgpuField,                                              \
    type##FvsPatchSphericalTensorgpuField                                         \
);                                                                             \
makeFvsPatchTypeField(fvsPatchSymmTensorgpuField, type##FvsPatchSymmTensorgpuField); \
makeFvsPatchTypeField(fvsPatchTensorgpuField, type##FvsPatchTensorgpuField);


#define makeFvsPatchTypeFieldTypedefs(type)                                    \
                                                                               \
typedef type##FvsPatchgpuField<scalar> type##FvsPatchScalargpuField;                 \
typedef type##FvsPatchgpuField<vector> type##FvsPatchVectorgpuField;                 \
typedef type##FvsPatchgpuField<sphericalTensor>                                   \
    type##FvsPatchSphericalTensorgpuField;                                        \
typedef type##FvsPatchgpuField<symmTensor> type##FvsPatchSymmTensorgpuField;         \
typedef type##FvsPatchgpuField<tensor> type##FvsPatchTensorgpuField;


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
