/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2015-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "fixedFluxPressureFvPatchScalargpuField.H"
#include "fvPatchgpuFieldMapper.H"
#include "volgpuFields.H"
#include "surfacegpuFields.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::fixedFluxPressureFvPatchScalargpuField::fixedFluxPressureFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedGradientFvPatchScalargpuField(p, iF),
    curTimeIndex_(-1)
{}


Foam::fixedFluxPressureFvPatchScalargpuField::fixedFluxPressureFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    fixedGradientFvPatchScalargpuField(p, iF),
    curTimeIndex_(-1)
{
    patchType() = dict.getOrDefault<word>("patchType", word::null);
    if (dict.found("value") && dict.found("gradient"))
    {
        fvPatchgpuField<scalar>::operator=
        (
            scalargpuField("value", dict, p.size())
        );
        gradient() = scalargpuField("gradient", dict, p.size());
    }
    else
    {
        fvPatchgpuField<scalar>::operator=(patchInternalField());
        gradient() = 0.0;
    }
}


Foam::fixedFluxPressureFvPatchScalargpuField::fixedFluxPressureFvPatchScalargpuField
(
    const fixedFluxPressureFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    fixedGradientFvPatchScalargpuField(p, iF),
    curTimeIndex_(-1)
{
    patchType() = ptf.patchType();

    // Map gradient. Set unmapped values and overwrite with mapped ptf
    gradient() = 0.0;
    gradient().map(ptf.gradient(), mapper);

    // Evaluate the value field from the gradient if the internal field is valid
    if (notNull(iF))
    {
        if (iF.size())
        {
            // Note: cannot ask for nf() if zero faces

            scalargpuField::operator=
            (
                //patchInternalField() + gradient()/patch().deltaCoeffs()
                // ***HGW Hack to avoid the construction of mesh.deltaCoeffs
                // which fails for AMI patches for some mapping operations
                patchInternalField()
              + gradient()*(patch().nf() & patch().delta())
            );
        }
    }
    else
    {
        // Enforce mapping of values so we have a valid starting value
        this->map(ptf, mapper);
    }
}


Foam::fixedFluxPressureFvPatchScalargpuField::fixedFluxPressureFvPatchScalargpuField
(
    const fixedFluxPressureFvPatchScalargpuField& wbppsf
)
:
    fixedGradientFvPatchScalargpuField(wbppsf),
    curTimeIndex_(-1)
{}


Foam::fixedFluxPressureFvPatchScalargpuField::fixedFluxPressureFvPatchScalargpuField
(
    const fixedFluxPressureFvPatchScalargpuField& wbppsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedGradientFvPatchScalargpuField(wbppsf, iF),
    curTimeIndex_(-1)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::fixedFluxPressureFvPatchScalargpuField::updateSnGrad
(
    const scalargpuField& snGradp
)
{
    if (updated())
    {
        return;
    }

    curTimeIndex_ = this->db().time().timeIndex();

    gradient() = snGradp;
    fixedGradientFvPatchScalargpuField::updateCoeffs();
}


void Foam::fixedFluxPressureFvPatchScalargpuField::updateCoeffs()
{
    if (updated())
    {
        return;
    }

    if (curTimeIndex_ != this->db().time().timeIndex())
    {
        FatalErrorInFunction
            << "updateCoeffs(const scalarField& snGradp) MUST be called before"
               " updateCoeffs() or evaluate() to set the boundary gradient."
            << exit(FatalError);
    }
}


void Foam::fixedFluxPressureFvPatchScalargpuField::write(Ostream& os) const
{
    fixedGradientFvPatchScalargpuField::write(os);
    writeEntry("value", os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        fixedFluxPressureFvPatchScalargpuField
    );
}


// ************************************************************************* //
