/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvMatrix.H"
#include "volgpuFields.H"
#include "cyclicAMIFvPatchgpuField.H"
#include "gpulduAddressingFunctors.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Type>
Foam::cyclicAMIFvPatchgpuField<Type>::cyclicAMIFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    cyclicAMILduInterfacegpuField(),
    coupledFvPatchgpuField<Type>(p, iF),
    cyclicAMIPatch_(refCast<const cyclicAMIgpuFvPatch>(p))
{}


template<class Type>
Foam::cyclicAMIFvPatchgpuField<Type>::cyclicAMIFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    cyclicAMILduInterfacegpuField(),
    coupledFvPatchgpuField<Type>(p, iF, dict, dict.found("value")),
    cyclicAMIPatch_(refCast<const cyclicAMIgpuFvPatch>(p, dict))
{
    if (!isA<cyclicAMIgpuFvPatch>(p))
    {
        FatalIOErrorInFunction(dict)
            << "    patch type '" << p.type()
            << "' not constraint type '" << typeName << "'"
            << "\n    for patch " << p.name()
            << " of field " << this->internalField().name()
            << " in file " << this->internalField().objectPath()
            << exit(FatalIOError);
    }

    if (!dict.found("value"))
    {
        if (this->coupled())
        {
            this->evaluate(Pstream::commsTypes::blocking);
        }
        else
        {
            fvPatchgpuField<Type>::operator=(this->patchInternalField());
        }
    }
}


template<class Type>
Foam::cyclicAMIFvPatchgpuField<Type>::cyclicAMIFvPatchgpuField
(
    const cyclicAMIFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    cyclicAMILduInterfacegpuField(),
    coupledFvPatchgpuField<Type>(ptf, p, iF, mapper),
    cyclicAMIPatch_(refCast<const cyclicAMIgpuFvPatch>(p))
{
    if (!isA<cyclicAMIgpuFvPatch>(this->patch()))
    {
        FatalErrorInFunction
            << "' not constraint type '" << typeName << "'"
            << "\n    for patch " << p.name()
            << " of field " << this->internalField().name()
            << " in file " << this->internalField().objectPath()
            << exit(FatalError);
    }
}


template<class Type>
Foam::cyclicAMIFvPatchgpuField<Type>::cyclicAMIFvPatchgpuField
(
    const cyclicAMIFvPatchgpuField<Type>& ptf
)
:
    cyclicAMILduInterfacegpuField(),
    coupledFvPatchgpuField<Type>(ptf),
    cyclicAMIPatch_(ptf.cyclicAMIPatch_)
{}


template<class Type>
Foam::cyclicAMIFvPatchgpuField<Type>::cyclicAMIFvPatchgpuField
(
    const cyclicAMIFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    cyclicAMILduInterfacegpuField(),
    coupledFvPatchgpuField<Type>(ptf, iF),
    cyclicAMIPatch_(ptf.cyclicAMIPatch_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
bool Foam::cyclicAMIFvPatchgpuField<Type>::coupled() const
{
    return cyclicAMIPatch_.coupled();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::cyclicAMIFvPatchgpuField<Type>::patchNeighbourField() const
{
    const gpuField<Type>& iField = this->primitiveField();

    // By pass polyPatch to get nbrId. Instead use cyclicAMIFvPatch virtual
    // neighbPatch()
    const cyclicAMIgpuFvPatch& neighbPatch = cyclicAMIPatch_.neighbPatch();
    const labelgpuList& nbrFaceCells = neighbPatch.gpuFaceCells();

    gpuField<Type> pnf(iField, nbrFaceCells);

    tmp<gpuField<Type>> tpnf;
    if (cyclicAMIPatch_.applyLowWeightCorrection())
    {
        gpuField<Type> pnfInternal(iField, cyclicAMIPatch_.gpuFaceCells());

        tpnf = cyclicAMIPatch_.interpolate(pnf, pnfInternal);
    }
    else
    {
        tpnf = cyclicAMIPatch_.interpolate(pnf);
    }

    if (doTransform())
    {
        tpnf.ref() = transform(gpuForwardT(), tpnf());
    }

    return tpnf;
}


template<class Type>
const Foam::cyclicAMIFvPatchgpuField<Type>&
Foam::cyclicAMIFvPatchgpuField<Type>::neighbourPatchField() const
{
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& fld =
        static_cast<const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&>
        (
            this->primitiveField()
        );

    return refCast<const cyclicAMIFvPatchgpuField<Type>>
    (
        fld.boundaryField()[cyclicAMIPatch_.neighbPatchID()]
    );
}


template<class Type>
void Foam::cyclicAMIFvPatchgpuField<Type>::updateInterfaceMatrix
(
    scalargpuField& result,
    const bool add,
    const gpulduAddressing& lduAddr,
    const label patchId,
    const scalargpuField& psiInternal,
    const scalargpuField& coeffs,
    const direction cmpt,
    const Pstream::commsTypes
) const
{
    const labelgpuList& nbrFaceCells =
        lduAddr.patchAddr(cyclicAMIPatch_.neighbPatchID());

    scalargpuField pnf(psiInternal, nbrFaceCells);

    const labelgpuList& faceCells = lduAddr.patchAddr(patchId);

    // Transform according to the transformation tensors
    transformCoupleField(pnf, cmpt);

    if (cyclicAMIPatch_.applyLowWeightCorrection())
    {
        scalargpuField pif(psiInternal, faceCells);
        pnf = cyclicAMIPatch_.interpolate(pnf, pif);
    }
    else
    {
        pnf = cyclicAMIPatch_.interpolate(pnf);
    }

    // Multiply the field by coefficients and add into the result
    coupledFvPatchgpuField<Type>::updateInterfaceMatrix(result, coeffs, pnf, !add);
}

template<class Type>
void Foam::cyclicAMIFvPatchgpuField<Type>::updateInterfaceMatrix
(
    gpuField<Type>& result,
    const bool add,
    const gpulduAddressing& lduAddr,
    const label patchId,
    const gpuField<Type>& psiInternal,
    const scalargpuField& coeffs,
    const Pstream::commsTypes
) const
{
    const labelgpuList& nbrFaceCells =
        lduAddr.patchAddr(cyclicAMIPatch_.neighbPatchID());

    gpuField<Type> pnf(psiInternal, nbrFaceCells);

    // Transform according to the transformation tensors
    transformCoupleField(pnf);

    if (cyclicAMIPatch_.applyLowWeightCorrection())
    {
        gpuField<Type> pif(psiInternal, cyclicAMIPatch_.gpuFaceCells());
        pnf = cyclicAMIPatch_.interpolate(pnf, pif);
    }
    else
    {
        pnf = cyclicAMIPatch_.interpolate(pnf);
    }

    // Multiply the field by coefficients and add into the result
    this->addToInternalField(result, !add, patchId, lduAddr, coeffs, pnf);
}


template<class Type>
void Foam::cyclicAMIFvPatchgpuField<Type>::manipulateMatrix
(
    gpufvMatrix<Type>& matrix,
    const label mat,
    const direction cmpt
)
{

    if (this->cyclicAMIPatch().owner())
    {
        label index = this->patch().index();

        const label globalPatchID =
            matrix.lduMeshAssembly().patchLocalToGlobalMap()[mat][index];

        const gpuField<scalar> intCoeffsCmpt
        (
            matrix.internalCoeffs()[globalPatchID].component(cmpt)
        );

        const gpuField<scalar> boundCoeffsCmpt
        (
            matrix.boundaryCoeffs()[globalPatchID].component(cmpt)
        );

        tmp<gpuField<scalar>> tintCoeffs(coeffs(matrix, intCoeffsCmpt, mat));
        tmp<gpuField<scalar>> tbndCoeffs(coeffs(matrix, boundCoeffsCmpt, mat));
        const gpuField<scalar>& intCoeffs = tintCoeffs.ref();
        const gpuField<scalar>& bndCoeffs = tbndCoeffs.ref();

        const labelgpuList& u = matrix.lduAddr().upperAddr();
        const labelgpuList& l = matrix.lduAddr().lowerAddr();

        label subFaceI = 0;

        const labelList& faceMap =
            matrix.lduMeshAssembly().faceBoundMap()[mat][index];

        forAll (faceMap, j)
        {
            label globalFaceI = faceMap[j];

            const scalar boundCorr = -bndCoeffs.get(subFaceI);
            const scalar intCorr = -intCoeffs.get(subFaceI);

            const scalar vu =  matrix.gpuUpper().get(globalFaceI);
            const scalar vdu = matrix.gpuDiag().get(u.get(globalFaceI));
            const scalar vdl = matrix.gpuDiag().get(l.get(globalFaceI));

            matrix.gpuUpper().set(globalFaceI, vu + boundCorr);
            matrix.gpuDiag().set(u.get(globalFaceI), vdu - intCorr);
            matrix.gpuDiag().set(l.get(globalFaceI), vdl - boundCorr);

            if (matrix.asymmetric())
            {
                const scalar vl =  matrix.gpuLower().get(globalFaceI);
                matrix.gpuLower().set(globalFaceI, vl + intCorr);
            }

            subFaceI++;
        }

        // Set internalCoeffs and boundaryCoeffs in the assembly matrix
        // on clyclicAMI patches to be used in the individual matrix by
        // matrix.flux()
        if (matrix.psi(mat).mesh().hostmesh().fluxRequired(this->internalField().name()))
        {
            matrix.internalCoeffs().set
            (
                globalPatchID, intCoeffs*pTraits<Type>::one
            );
            matrix.boundaryCoeffs().set
            (
                globalPatchID, bndCoeffs*pTraits<Type>::one
            );

            const label nbrPathID =
                cyclicAMIPatch_.cyclicAMIPatch().neighbPatchID();

            const label nbrGlobalPatchID =
                matrix.lduMeshAssembly().patchLocalToGlobalMap()
                    [mat][nbrPathID];

            matrix.internalCoeffs().set
            (
                nbrGlobalPatchID, intCoeffs*pTraits<Type>::one
            );
            matrix.boundaryCoeffs().set
            (
                nbrGlobalPatchID, bndCoeffs*pTraits<Type>::one
            );
        }
    }
}


template<class Type>
Foam::tmp<Foam::gpuField<Foam::scalar>>
Foam::cyclicAMIFvPatchgpuField<Type>::coeffs
(
    gpufvMatrix<Type>& matrix,
    const gpuField<scalar>& coeffs,
    const label mat
) const
{
    const label index(this->patch().index());

    const label nSubFaces
    (
        matrix.lduMeshAssembly().cellBoundMap()[mat][index].size()
    );

    gpuField<scalar> mapCoeffs(nSubFaces, Zero);

    const scalarListList& srcWeight =
        cyclicAMIPatch_.cyclicAMIPatch().AMI().srcWeights();

    label subFaceI = 0;
    forAll(*this, faceI)
    {
        const scalarList& w = srcWeight[faceI];
        for(label i=0; i<w.size(); i++)
        {
            const label localFaceId =
                matrix.lduMeshAssembly().facePatchFaceMap()[mat][index][subFaceI];
            mapCoeffs.set(subFaceI, w[i]*coeffs.get(localFaceId));
            subFaceI++;
        }
    }

    return tmp<gpuField<scalar>>(new gpuField<scalar>(mapCoeffs));
}


template<class Type>
template<class Type2>
void Foam::cyclicAMIFvPatchgpuField<Type>::collectStencilData
(
    const refPtr<mapDistribute>& mapPtr,
    const labelListList& stencil,
    const Type2& data,
    List<Type2>& expandedData
)
{
    expandedData.setSize(stencil.size());
    if (mapPtr.valid())
    {
        Type2 work(data);
        mapPtr().distribute(work);

        forAll(stencil, facei)
        {
            const labelList& slots = stencil[facei];
            expandedData[facei].append
            (
                UIndirectList<typename Type2::value_type>(work, slots)
            );
        }
    }
    else
    {
        forAll(stencil, facei)
        {
            const labelList& slots = stencil[facei];
            expandedData[facei].append
            (
                UIndirectList<typename Type2::value_type>(data, slots)
            );
        }
    }
}


template<class Type>
void Foam::cyclicAMIFvPatchgpuField<Type>::write(Ostream& os) const
{
    fvPatchgpuField<Type>::write(os);
    this->writeEntry("value", os);
}


// ************************************************************************* //

