/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::mixedFvPatchgpuField

Group
    grpGenericBoundaryConditions

Description
    This boundary condition provides a base class for 'mixed' type boundary
    conditions, i.e. conditions that mix fixed value and patch-normal gradient
    conditions.

    The respective contributions from each is determined by a weight field:

        \f[
            x_p = w x_p + (1-w) \left(x_c + \frac{\nabla_\perp x}{\Delta}\right)
        \f]

    where
    \vartable
        x_p   | patch values
        x_c   | patch internal cell values
        \Delta| inverse distance from face centre to internal cell centre
        w     | weighting values (0-1)
    \endvartable


Usage
    \table
        Property     | Description                          | Required | Default
        refValue     | fixed value                          | yes |
        refGradient  | patch normal gradient                | yes |
        valueFraction | value weighting (0-1)               | yes |
    \endtable

Note
    This condition is not usually applied directly; instead, use a derived
    mixed condition such as \c inletOutlet

See also
    Foam::inletOutletFvPatchField

SourceFiles
    mixedFvPatchgpuField.C

\*---------------------------------------------------------------------------*/

#ifndef mixedFvPatchgpuField_H
#define mixedFvPatchgpuField_H

#include "fvPatchgpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class mixedFvPatchgpuField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class mixedFvPatchgpuField
:
    public fvPatchgpuField<Type>
{
    // Private data

        //- Value field
        gpuField<Type> refValue_;

        //- Normal gradient field
        gpuField<Type> refGrad_;

        //- Fraction (0-1) of value used for boundary condition
        scalargpuField valueFraction_;

        //- Source field
        gpuField<Type> source_;


public:

    //- Runtime type information
    TypeName("mixed");


    // Constructors

        //- Construct from patch and internal field
        mixedFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct from patch, internal field and dictionary
        mixedFvPatchgpuField
        (
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const dictionary&
        );

        //- Construct by mapping the given mixedFvPatchgpuField onto a new patch
        mixedFvPatchgpuField
        (
            const mixedFvPatchgpuField<Type>&,
            const gpufvPatch&,
            const DimensionedgpuField<Type, gpuvolMesh>&,
            const fvPatchgpuFieldMapper&
        );

        //- Construct as copy
        mixedFvPatchgpuField
        (
            const mixedFvPatchgpuField<Type>&
        );

        //- Construct and return a clone
        virtual tmp<fvPatchgpuField<Type>> clone() const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new mixedFvPatchgpuField<Type>(*this)
            );
        }

        //- Construct as copy setting internal field reference
        mixedFvPatchgpuField
        (
            const mixedFvPatchgpuField<Type>&,
            const DimensionedgpuField<Type, gpuvolMesh>&
        );

        //- Construct and return a clone setting internal field reference
        virtual tmp<fvPatchgpuField<Type>> clone
        (
            const DimensionedgpuField<Type, gpuvolMesh>& iF
        ) const
        {
            return tmp<fvPatchgpuField<Type>>
            (
                new mixedFvPatchgpuField<Type>(*this, iF)
            );
        }


    // Member functions

        // Access

            //- Return true if this patch field fixes a value.
            //  Needed to check if a level has to be specified while solving
            //  Poissons equations.
            virtual bool fixesValue() const
            {
                return true;
            }

            //- Return false: this patch field is not altered by assignment
            virtual bool assignable() const
            {
                return false;
            }


        // Return defining fields

            virtual gpuField<Type>& refValue()
            {
                return refValue_;
            }

            virtual const gpuField<Type>& refValue() const
            {
                return refValue_;
            }

            virtual gpuField<Type>& refGrad()
            {
                return refGrad_;
            }

            virtual const gpuField<Type>& refGrad() const
            {
                return refGrad_;
            }

            virtual scalargpuField& valueFraction()
            {
                return valueFraction_;
            }

            virtual const scalargpuField& valueFraction() const
            {
                return valueFraction_;
            }

            virtual gpuField<Type>& source()
            {
                return source_;
            }

            virtual const gpuField<Type>& source() const
            {
                return source_;
            }


        // Mapping functions

            //- Map (and resize as needed) from self given a mapping object
            virtual void autoMap
            (
                const fvPatchgpuFieldMapper&
            );

            //- Reverse map the given fvPatchgpuField onto this fvPatchgpuField
            virtual void rmap
            (
                const fvPatchgpuField<Type>&,
                const labelgpuList&
            );


        // Evaluation functions

            //- Return gradient at boundary
            virtual tmp<gpuField<Type>> snGrad() const;

            //- Evaluate the patch field
            virtual void evaluate
            (
                const Pstream::commsTypes commsType =
                    Pstream::commsTypes::blocking
            );

            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the value of this patchField with given weights
            virtual tmp<gpuField<Type>> valueInternalCoeffs
            (
                const tmp<scalargpuField>&
            ) const;

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the value of this patchField with given weights
            virtual tmp<gpuField<Type>> valueBoundaryCoeffs
            (
                const tmp<scalargpuField>&
            ) const;

            //- Return the matrix diagonal coefficients corresponding to the
            //  evaluation of the gradient of this patchField
            virtual tmp<gpuField<Type>> gradientInternalCoeffs() const;

            //- Return the matrix source coefficients corresponding to the
            //  evaluation of the gradient of this patchField
            virtual tmp<gpuField<Type>> gradientBoundaryCoeffs() const;


        //- Write
        virtual void write(Ostream&) const;


    // Member operators

        virtual void operator=(const UList<Type>&) {}
        virtual void operator=(const gpuList<Type>&) {}

        virtual void operator=(const fvPatchgpuField<Type>&) {}
        virtual void operator+=(const fvPatchgpuField<Type>&) {}
        virtual void operator-=(const fvPatchgpuField<Type>&) {}
        virtual void operator*=(const fvPatchgpuField<scalar>&) {}
        virtual void operator/=(const fvPatchgpuField<scalar>&) {}

        virtual void operator+=(const gpuField<Type>&) {}
        virtual void operator-=(const gpuField<Type>&) {}

        virtual void operator*=(const gpuField<scalar>&) {}
        virtual void operator/=(const gpuField<scalar>&) {}

        virtual void operator=(const Type&) {}
        virtual void operator+=(const Type&) {}
        virtual void operator-=(const Type&) {}
        virtual void operator*=(const scalar) {}
        virtual void operator/=(const scalar) {}
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "mixedFvPatchgpuField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
