/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "mixedFvPatchgpuField.H"

// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
Foam::mixedFvPatchgpuField<Type>::mixedFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    fvPatchgpuField<Type>(p, iF),
    refValue_(p.size()),
    refGrad_(p.size()),
    valueFraction_(p.size()),
    source_(p.size(), Zero)
{}


template<class Type>
Foam::mixedFvPatchgpuField<Type>::mixedFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    fvPatchgpuField<Type>(p, iF, dict, false),
    refValue_("refValue", dict, p.size()),
    refGrad_("refGradient", dict, p.size()),
    valueFraction_("valueFraction", dict, p.size()),
    source_(p.size(), Zero)
{
    // Could also check/clip fraction to 0-1 range
    evaluate();
}


template<class Type>
Foam::mixedFvPatchgpuField<Type>::mixedFvPatchgpuField
(
    const mixedFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    fvPatchgpuField<Type>(ptf, p, iF, mapper),
    refValue_(ptf.refValue_, mapper),
    refGrad_(ptf.refGrad_, mapper),
    valueFraction_(ptf.valueFraction_, mapper),
    source_(ptf.source_, mapper)
{
    if (notNull(iF) && mapper.hasUnmapped())
    {
        WarningInFunction
            << "On field " << iF.name() << " patch " << p.name()
            << " patchField " << this->type()
            << " : mapper does not map all values." << nl
            << "    To avoid this warning fully specify the mapping in derived"
            << " patch fields." << endl;
    }
}


template<class Type>
Foam::mixedFvPatchgpuField<Type>::mixedFvPatchgpuField
(
    const mixedFvPatchgpuField<Type>& ptf
)
:
    fvPatchgpuField<Type>(ptf),
    refValue_(ptf.refValue_),
    refGrad_(ptf.refGrad_),
    valueFraction_(ptf.valueFraction_),
    source_(ptf.source_)
{}


template<class Type>
Foam::mixedFvPatchgpuField<Type>::mixedFvPatchgpuField
(
    const mixedFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    fvPatchgpuField<Type>(ptf, iF),
    refValue_(ptf.refValue_),
    refGrad_(ptf.refGrad_),
    valueFraction_(ptf.valueFraction_),
    source_(ptf.source_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
void Foam::mixedFvPatchgpuField<Type>::autoMap
(
    const fvPatchgpuFieldMapper& m
)
{
    fvPatchgpuField<Type>::autoMap(m);
    refValue_.autoMap(m);
    refGrad_.autoMap(m);
    valueFraction_.autoMap(m);
    source_.autoMap(m);
}


template<class Type>
void Foam::mixedFvPatchgpuField<Type>::rmap
(
    const fvPatchgpuField<Type>& ptf,
    const labelgpuList& addr
)
{
    fvPatchgpuField<Type>::rmap(ptf, addr);

    const mixedFvPatchgpuField<Type>& mptf =
        refCast<const mixedFvPatchgpuField<Type>>(ptf);

    refValue_.rmap(mptf.refValue_, addr);
    refGrad_.rmap(mptf.refGrad_, addr);
    valueFraction_.rmap(mptf.valueFraction_, addr);
    source_.rmap(mptf.source_, addr);
}


template<class Type>
void Foam::mixedFvPatchgpuField<Type>::evaluate(const Pstream::commsTypes)
{

    if (!this->updated())
    {
        this->updateCoeffs();
    }

    gpuField<Type>::operator=
    (
        valueFraction_*refValue_
      + (1.0 - valueFraction_)
       *(
            this->patchInternalField()
          + refGrad_/this->patch().deltaCoeffs()
        )
    );

    fvPatchgpuField<Type>::evaluate();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::mixedFvPatchgpuField<Type>::snGrad() const
{
    
    return
        valueFraction_
        *(refValue_ - this->patchInternalField())
        *this->patch().deltaCoeffs()
        + (1.0 - valueFraction_)*refGrad_;
        
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::mixedFvPatchgpuField<Type>::valueInternalCoeffs
(
    const tmp<scalargpuField>&
) const
{
    return Type(pTraits<Type>::one)*(1.0 - valueFraction_);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::mixedFvPatchgpuField<Type>::valueBoundaryCoeffs
(
    const tmp<scalargpuField>&
) const
{
    return valueFraction_ * refValue_
           //+ (1.0 - valueFraction_)*refGrad_/this->patch().deltaCoeffs();
           + 1.0 / this->patch().deltaCoeffs() * refGrad_ * (1.0 - valueFraction_);
}

template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::mixedFvPatchgpuField<Type>::gradientInternalCoeffs() const
{
    return -Type(pTraits<Type>::one)*valueFraction_*this->patch().deltaCoeffs();
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::mixedFvPatchgpuField<Type>::gradientBoundaryCoeffs() const
{
    return
        valueFraction_*this->patch().deltaCoeffs()*refValue_
      + (1.0 - valueFraction_)*refGrad_;
}


template<class Type>
void Foam::mixedFvPatchgpuField<Type>::write(Ostream& os) const
{
    fvPatchgpuField<Type>::write(os);
    refValue_.writeEntry("refValue", os);
    refGrad_.writeEntry("refGradient", os);
    valueFraction_.writeEntry("valueFraction", os);
    source_.writeEntry("source", os);
    this->writeEntry("value", os);
}


// ************************************************************************* //
