/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::orthogonalSnGrad

Group
    grpFvSnGradSchemes

Description
    Surface gradient scheme with no non-orthogonal correction.

Usage
    Minimal example by using \c system/fvSchemes:
    \verbatim
    snGradSchemes
    {
        snGrad(<term>)       orthogonal;
    }
    \endverbatim

Note
  - Interpolation weighting factors (i.e. delta coefficients) are based
    on the \c deltaCoeffs function rather than the \c nonOrthDeltaCoeffs
    function, which is used by the \c uncorrected scheme.

SourceFiles
    orthogonalSnGrad.C

\*---------------------------------------------------------------------------*/

#ifndef gpuorthogonalSnGrad_H
#define gpuorthogonalSnGrad_H

#include "gpusnGradScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                        Class orthogonalSnGrad Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpuorthogonalSnGrad
:
    public gpusnGradScheme<Type>
{
    // Private Member Functions

        //- No copy assignment
        void operator=(const gpuorthogonalSnGrad&) = delete;


public:

    //- Runtime type information
    TypeName("orthogonal");


    // Constructors

        //- Construct from mesh
        gpuorthogonalSnGrad(const gpufvMesh& mesh)
        :
            gpusnGradScheme<Type>(mesh)
        {}

        //- Construct from mesh and data stream
        gpuorthogonalSnGrad(const gpufvMesh& mesh, Istream&)
        :
            gpusnGradScheme<Type>(mesh)
        {}


    //- Destructor
    virtual ~gpuorthogonalSnGrad() = default;


    // Member Functions

        //- Return the interpolation weighting factors for the given field
        virtual tmp<surfaceScalargpuField> deltaCoeffs
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const
        {
            return this->mesh().deltaCoeffs();
        }

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const noexcept
        {
            return false;
        }

        //- Return the explicit correction to the orthogonalSnGrad
        //- for the given field
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuorthogonalSnGrad.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
