/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2018-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::gpudfsolverGrad

Group
    grpFvGradSchemes

Description

SourceFiles


\*---------------------------------------------------------------------------*/

#ifndef gpudfsolverGrad_H
#define gpudfsolverGrad_H

#include "gpugradScheme.H"
#include "gpusurfaceInterpolationScheme.H"
#include "gpulinear.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                       Class gaussGrad Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpudfsolverGrad
:
    public fv::gpugradScheme<Type>
{
    // Private Data

        //- Interpolation scheme
        tmp<gpusurfaceInterpolationScheme<Type>> tinterpScheme_;


    // Private Member Functions

        //- No copy construct
        gpudfsolverGrad(const gpudfsolverGrad&) = delete;

        //- No copy assignment
        void operator=(const gpudfsolverGrad&) = delete;


public:

    //- Runtime type information
    TypeName("dfsolver");


    // Constructors

        //- Construct from mesh
        gpudfsolverGrad(const gpufvMesh& mesh)
        :
            gpugradScheme<Type>(mesh),
            tinterpScheme_(new gpulinear<Type>(mesh))
        {}

        //- Construct from mesh and Istream
        gpudfsolverGrad(const gpufvMesh& mesh, Istream& is)
        :
            gpugradScheme<Type>(mesh),
            tinterpScheme_(nullptr)
        {
            if (is.eof())
            {
                tinterpScheme_ =
                    tmp<gpusurfaceInterpolationScheme<Type>>
                    (
                        new gpulinear<Type>(mesh)
                    );
            }
            else
            {
                tinterpScheme_ =
                    tmp<gpusurfaceInterpolationScheme<Type>>
                    (
                        gpusurfaceInterpolationScheme<Type>::New(mesh, is)
                    );
            }
        }


    // Member Functions

        //- Return the gradient of the given field
        static
        tmp
        <
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > gradf
        (
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>&,
            const word& name
        );

        //- Return the gradient of the given field to the gradScheme::grad
        //- for optional caching
        virtual tmp
        <
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > calcGrad
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vsf,
            const word& name
        ) const;

        //- Correct the boundary values of the gradient using the patchField
        //- snGrad functions
        static void correctBoundaryConditions
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            GeometricgpuField
            <typename outerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>&
        );
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpudfsolverGrad.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
