/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvcMeshPhi.H"
#include "gpufvMesh.H"
#include "gpuddtScheme.H"
#include "gpusurfaceInterpolate.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::meshPhi
(
    const volVectorgpuField& vf
)
{
    return fv::gpuddtScheme<vector>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme("ddt(" + vf.name() + ')')
    ).ref().meshPhi(vf);
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::meshPhi
(
    const dimensionedScalar& rho,
    const volVectorgpuField& vf
)
{
    return fv::gpuddtScheme<vector>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme("ddt(" + rho.name() + ',' + vf.name() + ')')
    ).ref().meshPhi(vf);
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::meshPhi
(
    const volScalargpuField& rho,
    const volVectorgpuField& vf
)
{
    return fv::gpuddtScheme<vector>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme("ddt(" + rho.name() + ',' + vf.name() + ')')
    ).ref().meshPhi(vf);
}


void Foam::fvc::makeRelative
(
    surfaceScalargpuField& phi,
    const volVectorgpuField& U
)
{
    if (phi.mesh().hostmesh().moving())
    {
        phi -= fvc::meshPhi(U);
    }
}

void Foam::fvc::makeRelative
(
    surfaceScalargpuField& phi,
    const dimensionedScalar& rho,
    const volVectorgpuField& U
)
{
    if (phi.mesh().hostmesh().moving())
    {
        phi -= rho*fvc::meshPhi(rho, U);
    }
}

void Foam::fvc::makeRelative
(
    surfaceScalargpuField& phi,
    const volScalargpuField& rho,
    const volVectorgpuField& U
)
{
    if (phi.mesh().hostmesh().moving())
    {
        phi -= fvc::interpolate(rho)*fvc::meshPhi(rho, U);
    }
}


void Foam::fvc::makeAbsolute
(
    surfaceScalargpuField& phi,
    const volVectorgpuField& U
)
{
    if (phi.mesh().hostmesh().moving())
    {
        phi += fvc::meshPhi(U);
    }
}

void Foam::fvc::makeAbsolute
(
    surfaceScalargpuField& phi,
    const dimensionedScalar& rho,
    const volVectorgpuField& U
)
{
    if (phi.mesh().hostmesh().moving())
    {
        phi += rho*fvc::meshPhi(rho, U);
    }
}

void Foam::fvc::makeAbsolute
(
    surfaceScalargpuField& phi,
    const volScalargpuField& rho,
    const volVectorgpuField& U
)
{
    if (phi.mesh().hostmesh().moving())
    {
        phi += fvc::interpolate(rho)*fvc::meshPhi(rho, U);
    }
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::relative
(
    const tmp<surfaceScalargpuField>& tphi,
    const volVectorgpuField& U
)
{
    if (tphi().mesh().hostmesh().moving())
    {
        return tphi - fvc::meshPhi(U);
    }
    else
    {
        return tmp<surfaceScalargpuField>(tphi, true);
    }
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::relative
(
    const tmp<surfaceScalargpuField>& tphi,
    const volScalargpuField& rho,
    const volVectorgpuField& U
)
{
    if (tphi().mesh().hostmesh().moving())
    {
        return tphi - fvc::interpolate(rho)*fvc::meshPhi(rho, U);
    }
    else
    {
        return tmp<surfaceScalargpuField>(tphi, true);
    }
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::absolute
(
    const tmp<surfaceScalargpuField>& tphi,
    const volVectorgpuField& U
)
{
    if (tphi().mesh().hostmesh().moving())
    {
        return tphi + fvc::meshPhi(U);
    }
    else
    {
        return tmp<surfaceScalargpuField>(tphi, true);
    }
}


Foam::tmp<Foam::surfaceScalargpuField> Foam::fvc::absolute
(
    const tmp<surfaceScalargpuField>& tphi,
    const volScalargpuField& rho,
    const volVectorgpuField& U
)
{
    if (tphi().mesh().hostmesh().moving())
    {
        return tphi + fvc::interpolate(rho)*fvc::meshPhi(rho, U);
    }
    else
    {
        return tmp<surfaceScalargpuField>(tphi, true);
    }
}


void Foam::fvc::correctUf
(
    autoPtr<surfaceVectorgpuField>& Uf,
    const volVectorgpuField& U,
    const surfaceScalargpuField& phi
)
{
    const gpufvMesh& mesh = U.mesh();

    if (mesh.hostmesh().dynamic())
    {
        Uf() = fvc::interpolate(U);
        surfaceVectorgpuField n(mesh.Sf()/mesh.magSf());
        Uf() += n*(phi/mesh.magSf() - (n & Uf()));
    }
}


void Foam::fvc::correctRhoUf
(
    autoPtr<surfaceVectorgpuField>& rhoUf,
    const volScalargpuField& rho,
    const volVectorgpuField& U,
    const surfaceScalargpuField& phi
)
{
    const gpufvMesh& mesh = U.mesh();

    if (mesh.hostmesh().dynamic())
    {
        rhoUf() = fvc::interpolate(rho*U);
        surfaceVectorgpuField n(mesh.Sf()/mesh.magSf());
        rhoUf() += n*(fvc::absolute(phi, rho, U)/mesh.magSf() - (n & rhoUf()));
    }
}


// ************************************************************************* //
