/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::divScheme

Group
    grpFvDivSchemes

Description
    Abstract base class for div schemes.

SourceFiles
    divScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpudivScheme_H
#define gpudivScheme_H

#include "tmp.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "gpulinear.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

template<class Type>
class gpufvMatrix;

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                           Class divScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpudivScheme
:
    public refCount
{

protected:

    // Protected data

        const gpufvMesh& mesh_;
        tmp<gpusurfaceInterpolationScheme<Type>> tinterpScheme_;


    // Private Member Functions

        //- No copy construct
        gpudivScheme(const gpudivScheme&) = delete;

        //- No copy assignment
        void operator=(const gpudivScheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpudivScheme,
            Istream,
            (const gpufvMesh& mesh, Istream& schemeData),
            (mesh, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpudivScheme(const gpufvMesh& mesh)
        :
            mesh_(mesh),
            tinterpScheme_(new gpulinear<Type>(mesh))
        {}

        //- Construct from mesh and Istream
        gpudivScheme(const gpufvMesh& mesh, Istream& is)
        :
            mesh_(mesh),
            tinterpScheme_(gpusurfaceInterpolationScheme<Type>::New(mesh, is))
        {}


    // Selectors

        //- Return a pointer to a new divScheme created on freestore
        static tmp<gpudivScheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpudivScheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        virtual tmp
        <
            GeometricgpuField
            <typename innerProduct<vector, Type>::type, fvPatchgpuField, gpuvolMesh>
        > fvcDiv
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeFvgpuDivTypeScheme(SS, Type)                                          \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpudivScheme<Type>::addIstreamConstructorToTable<SS<Type>>            \
                add##SS##Type##IstreamConstructorToTable_;                     \
        }                                                                      \
    }

#define makeFvgpuDivScheme(SS)                                                    \
                                                                               \
makeFvgpuDivTypeScheme(SS, vector)                                                \
makeFvgpuDivTypeScheme(SS, sphericalTensor)                                       \
makeFvgpuDivTypeScheme(SS, symmTensor)                                            \
makeFvgpuDivTypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpudivScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
