/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::localEulerDdtScheme

Group
    grpFvDdtSchemes

Description
    Local time-step first-order Euler implicit/explicit ddt.

    The reciprocal of the local time-step field is looked-up from the database.

    This scheme should only be used for steady-state computations using
    transient codes where local time-stepping is preferably to under-relaxation
    for transport consistency reasons.

See also
    Foam::fv::CoEulerDdtScheme

SourceFiles
    localEulerDdt.C
    localEulerDdtScheme.C
    localEulerDdtSchemes.C

\*---------------------------------------------------------------------------*/

#ifndef gpulocalEulerDdtScheme_H
#define gpulocalEulerDdtScheme_H

#include "gpuddtScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                       Class localEulerDdt Declaration
\*---------------------------------------------------------------------------*/

class gpulocalEulerDdt
{
public:

    //- Name of the reciprocal local time-step field
    static word rDeltaTName;

    //- Name of the reciprocal local face time-step field
    static word rDeltaTfName;

    //- Name of the reciprocal local sub-cycling time-step field
    static word rSubDeltaTName;


    // Constructors

        gpulocalEulerDdt()
        {}


    // Member Functions

        //- Return true if LTS is enabled
        static bool enabled(const gpufvMesh& mesh);

        //- Return the reciprocal of the local time-step
        //  looked-up from the objectRegistry
        static const volScalargpuField& localRDeltaT(const gpufvMesh& mesh);

        //- Return the reciprocal of the local face time-step
        //  looked-up from the objectRegistry
        static const surfaceScalargpuField& localRDeltaTf(const gpufvMesh& mesh);

        //- Calculate and return the reciprocal of the local sub-cycling
        //  time-step
        static tmp<volScalargpuField> localRSubDeltaT
        (
            const gpufvMesh& mesh,
            const label nAlphaSubCycles
        );
};


/*---------------------------------------------------------------------------*\
                       Class localEulerDdtScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulocalEulerDdtScheme
:
    public gpulocalEulerDdt,
    public fv::gpuddtScheme<Type>
{
    // Private Member Functions

        //- No copy construct
        gpulocalEulerDdtScheme(const gpulocalEulerDdtScheme&) = delete;

        //- No copy assignment
        void operator=(const gpulocalEulerDdtScheme&) = delete;

        //- Return the reciprocal of the local time-step
        const volScalargpuField& localRDeltaT() const;

        //- Return the reciprocal of the local face time-step
        const surfaceScalargpuField& localRDeltaTf() const;


public:

    //- Runtime type information
    TypeName("localEuler");


    // Constructors

        //- Construct from mesh
        gpulocalEulerDdtScheme(const gpufvMesh& mesh)
        :
            gpuddtScheme<Type>(mesh)
        {}

        //- Construct from mesh and Istream
        gpulocalEulerDdtScheme(const gpufvMesh& mesh, Istream& is)
        :
            gpuddtScheme<Type>(mesh, is)
        {}


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return fv::gpuddtScheme<Type>::mesh();
        }

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensioned<Type>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        typedef typename gpuddtScheme<Type>::fluxFieldType fluxFieldType;

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<surfaceScalargpuField> meshPhi
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );
};


template<>
tmp<surfaceScalargpuField> gpulocalEulerDdtScheme<scalar>::fvcDdtUfCorr
(
    const GeometricgpuField<scalar, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<scalar, fvsPatchgpuField, gpusurfaceMesh>& Uf
);

template<>
tmp<surfaceScalargpuField> gpulocalEulerDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);

template<>
tmp<surfaceScalargpuField> gpulocalEulerDdtScheme<scalar>::fvcDdtUfCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& Uf
);

template<>
tmp<surfaceScalargpuField> gpulocalEulerDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpulocalEulerDdtScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
