/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2018 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpulocalEulerDdtScheme.H"
#include "gpusurfaceInterpolate.H"
#include "gpufvMatrices.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type>
const volScalargpuField& gpulocalEulerDdtScheme<Type>::localRDeltaT() const
{
    return gpulocalEulerDdt::localRDeltaT(mesh());
}


template<class Type>
const surfaceScalargpuField& gpulocalEulerDdtScheme<Type>::localRDeltaTf() const
{
    return gpulocalEulerDdt::localRDeltaTf(mesh());
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpulocalEulerDdtScheme<Type>::fvcDdt
(
    const dimensioned<Type>& dt
)
{
    IOobject ddtIOobject
    (
        "ddt(" + dt.name() + ')',
        mesh().time().timeName(),
        mesh().hostmesh()
    );

    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>::New
    (
        ddtIOobject,
        mesh(),
        dimensioned<Type>(dt.dimensions()/dimTime, Zero),
        calculatedFvPatchgpuField<Type>::typeName
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpulocalEulerDdtScheme<Type>::fvcDdt
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const volScalargpuField& rDeltaT = localRDeltaT();

    IOobject ddtIOobject
    (
        "ddt(" + vf.name() + ')',
        mesh().time().timeName(),
        mesh().hostmesh()
    );

    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
    (
        new GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>
        (
            ddtIOobject,
            rDeltaT*(vf - vf.oldTime())
        )
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpulocalEulerDdtScheme<Type>::fvcDdt
(
    const dimensionedScalar& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const volScalargpuField& rDeltaT = localRDeltaT();

    IOobject ddtIOobject
    (
        "ddt(" + rho.name() + ',' + vf.name() + ')',
        mesh().time().timeName(),
        mesh().hostmesh()
    );

    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
    (
        new GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>
        (
            ddtIOobject,
            rDeltaT*rho*(vf - vf.oldTime())
        )
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpulocalEulerDdtScheme<Type>::fvcDdt
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const volScalargpuField& rDeltaT = localRDeltaT();

    IOobject ddtIOobject
    (
        "ddt(" + rho.name() + ',' + vf.name() + ')',
        mesh().time().timeName(),
        mesh().hostmesh()
    );

    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
    (
        new GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>
        (
            ddtIOobject,
            rDeltaT*(rho*vf - rho.oldTime()*vf.oldTime())
        )
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
gpulocalEulerDdtScheme<Type>::fvcDdt
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    const volScalargpuField& rDeltaT = localRDeltaT();

    IOobject ddtIOobject
    (
        "ddt("+alpha.name()+','+rho.name()+','+vf.name()+')',
        mesh().time().timeName(),
        mesh().hostmesh()
    );

    return tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
    (
        new GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>
        (
            ddtIOobject,
            rDeltaT
           *(
               alpha*rho*vf
             - alpha.oldTime()*rho.oldTime()*vf.oldTime()
           )
        )
    );
}


template<class Type>
tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
gpulocalEulerDdtScheme<Type>::fvcDdt
(
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& sf
)
{
    const surfaceScalargpuField& rDeltaT = localRDeltaTf();

    IOobject ddtIOobject
    (
        "ddt("+sf.name()+')',
        mesh().time().timeName(),
        mesh().hostmesh()
    );

    return tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
    (
        new GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>
        (
            ddtIOobject,
            rDeltaT*(sf - sf.oldTime())
        )
    );
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpulocalEulerDdtScheme<Type>::fvmDdt
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            vf.dimensions()*dimVol/dimTime
        )
    );

    gpufvMatrix<Type>& fvm = tfvm.ref();

    const scalargpuField& rDeltaT = localRDeltaT();

    fvm.gpuDiag() = rDeltaT*mesh().Vsc();
    fvm.source() = rDeltaT*vf.oldTime().primitiveField()*mesh().Vsc();

    return tfvm;
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpulocalEulerDdtScheme<Type>::fvmDdt
(
    const dimensionedScalar& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            rho.dimensions()*vf.dimensions()*dimVol/dimTime
        )
    );
    gpufvMatrix<Type>& fvm = tfvm.ref();

    const scalargpuField& rDeltaT = localRDeltaT();

    fvm.gpuDiag() = rDeltaT*rho.value()*mesh().Vsc();

    fvm.source() =
        rDeltaT*rho.value()*vf.oldTime().primitiveField()*mesh().Vsc();

    return tfvm;
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpulocalEulerDdtScheme<Type>::fvmDdt
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            rho.dimensions()*vf.dimensions()*dimVol/dimTime
        )
    );
    gpufvMatrix<Type>& fvm = tfvm.ref();

    const scalargpuField& rDeltaT = localRDeltaT();

    fvm.gpuDiag() = rDeltaT*rho.primitiveField()*mesh().Vsc();

    fvm.source() = rDeltaT
       *rho.oldTime().primitiveField()
       *vf.oldTime().primitiveField()*mesh().Vsc();

    return tfvm;
}


template<class Type>
tmp<gpufvMatrix<Type>>
gpulocalEulerDdtScheme<Type>::fvmDdt
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    tmp<gpufvMatrix<Type>> tfvm
    (
        new gpufvMatrix<Type>
        (
            vf,
            alpha.dimensions()*rho.dimensions()*vf.dimensions()*dimVol/dimTime
        )
    );
    gpufvMatrix<Type>& fvm = tfvm.ref();

    const scalargpuField& rDeltaT = localRDeltaT();

    fvm.gpuDiag() =
        rDeltaT*alpha.primitiveField()*rho.primitiveField()*mesh().Vsc();

    fvm.source() = rDeltaT
       *alpha.oldTime().primitiveField()
       *rho.oldTime().primitiveField()
       *vf.oldTime().primitiveField()*mesh().Vsc();

    return tfvm;
}


template<class Type>
tmp<typename gpulocalEulerDdtScheme<Type>::fluxFieldType>
gpulocalEulerDdtScheme<Type>::fvcDdtUfCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
)
{
    const surfaceScalargpuField rDeltaT(fvc::interpolate(localRDeltaT()));

    fluxFieldType phiUf0(mesh().Sf() & Uf.oldTime());
    fluxFieldType phiCorr
    (
        phiUf0 - fvc::dotInterpolate(mesh().Sf(), U.oldTime())
    );

    return tmp<fluxFieldType>
    (
        new fluxFieldType
        (
            IOobject
            (
                "ddtCorr(" + U.name() + ',' + Uf.name() + ')',
                mesh().time().timeName(),
                mesh().hostmesh()
            ),
            this->fvcDdtPhiCoeff(U.oldTime(), phiUf0, phiCorr)
           *rDeltaT*phiCorr
        )
    );
}


template<class Type>
tmp<typename gpulocalEulerDdtScheme<Type>::fluxFieldType>
gpulocalEulerDdtScheme<Type>::fvcDdtPhiCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const fluxFieldType& phi
)
{
    const surfaceScalargpuField rDeltaT(fvc::interpolate(localRDeltaT()));

    fluxFieldType phiCorr
    (
        phi.oldTime() - fvc::dotInterpolate(mesh().Sf(), U.oldTime())
    );

    return tmp<fluxFieldType>
    (
        new fluxFieldType
        (
            IOobject
            (
                "ddtCorr(" + U.name() + ',' + phi.name() + ')',
                mesh().time().timeName(),
                mesh().hostmesh()
            ),
            this->fvcDdtPhiCoeff(U.oldTime(), phi.oldTime(), phiCorr)
           *rDeltaT*phiCorr
        )
    );
}


template<class Type>
tmp<typename gpulocalEulerDdtScheme<Type>::fluxFieldType>
gpulocalEulerDdtScheme<Type>::fvcDdtUfCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
)
{
    const surfaceScalargpuField rDeltaT(fvc::interpolate(localRDeltaT()));

    if
    (
        U.dimensions() == dimVelocity
     && Uf.dimensions() == dimDensity*dimVelocity
    )
    {
        GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh> rhoU0
        (
            rho.oldTime()*U.oldTime()
        );

        fluxFieldType phiUf0(mesh().Sf() & Uf.oldTime());
        fluxFieldType phiCorr(phiUf0 - fvc::dotInterpolate(mesh().Sf(), rhoU0));

        return tmp<fluxFieldType>
        (
            new fluxFieldType
            (
                IOobject
                (
                    "ddtCorr("
                  + rho.name() + ',' + U.name() + ',' + Uf.name() + ')',
                    mesh().time().timeName(),
                    mesh().hostmesh()
                ),
                this->fvcDdtPhiCoeff(rhoU0, phiUf0, phiCorr, rho.oldTime())
               *rDeltaT*phiCorr
            )
        );
    }
    else if
    (
        U.dimensions() == dimDensity*dimVelocity
     && Uf.dimensions() == dimDensity*dimVelocity
    )
    {
        fluxFieldType phiUf0(mesh().Sf() & Uf.oldTime());
        fluxFieldType phiCorr
        (
            phiUf0 - fvc::dotInterpolate(mesh().Sf(), U.oldTime())
        );

        return tmp<fluxFieldType>
        (
            new fluxFieldType
            (
                IOobject
                (
                    "ddtCorr("
                  + rho.name() + ',' + U.name() + ',' + Uf.name() + ')',
                    mesh().time().timeName(),
                    mesh().hostmesh()
                ),
                this->fvcDdtPhiCoeff
                (
                    U.oldTime(),
                    phiUf0,
                    phiCorr,
                    rho.oldTime()
                )*rDeltaT*phiCorr
            )
        );
    }
    else
    {
        FatalErrorInFunction
            << "dimensions of Uf are not correct"
            << abort(FatalError);

        return fluxFieldType::null();
    }
}


template<class Type>
tmp<typename gpulocalEulerDdtScheme<Type>::fluxFieldType>
gpulocalEulerDdtScheme<Type>::fvcDdtPhiCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const fluxFieldType& phi
)
{
    const surfaceScalargpuField rDeltaT(fvc::interpolate(localRDeltaT()));

    if
    (
        U.dimensions() == dimVelocity
     && phi.dimensions() == rho.dimensions()*dimVelocity*dimArea
    )
    {
        GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh> rhoU0
        (
            rho.oldTime()*U.oldTime()
        );

        fluxFieldType phiCorr
        (
            phi.oldTime() - fvc::dotInterpolate(mesh().Sf(), rhoU0)
        );

        return tmp<fluxFieldType>
        (
            new fluxFieldType
            (
                IOobject
                (
                    "ddtCorr("
                  + rho.name() + ',' + U.name() + ',' + phi.name() + ')',
                    mesh().time().timeName(),
                    mesh().hostmesh()
                ),
                this->fvcDdtPhiCoeff
                (
                    rhoU0,
                    phi.oldTime(),
                    phiCorr,
                    rho.oldTime()
                )*rDeltaT*phiCorr
            )
        );
    }
    else if
    (
        U.dimensions() == rho.dimensions()*dimVelocity
     && phi.dimensions() == rho.dimensions()*dimVelocity*dimArea
    )
    {
        fluxFieldType phiCorr
        (
            phi.oldTime() - fvc::dotInterpolate(mesh().Sf(), U.oldTime())
        );

        return tmp<fluxFieldType>
        (
            new fluxFieldType
            (
                IOobject
                (
                    "ddtCorr("
                  + rho.name() + ',' + U.name() + ',' + phi.name() + ')',
                    mesh().time().timeName(),
                    mesh().hostmesh()
                ),
                this->fvcDdtPhiCoeff
                (
                    U.oldTime(),
                    phi.oldTime(),
                    phiCorr,
                    rho.oldTime()
                )*rDeltaT*phiCorr
            )
        );
    }
    else
    {
        FatalErrorInFunction
            << "dimensions of phi are not correct"
            << abort(FatalError);

        return fluxFieldType::null();
    }
}


template<class Type>
tmp<surfaceScalargpuField> gpulocalEulerDdtScheme<Type>::meshPhi
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
)
{
    tmp<surfaceScalargpuField> tmeshPhi
    (
        new surfaceScalargpuField
        (
            IOobject
            (
                "meshPhi",
                mesh().time().timeName(),
                mesh().hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE,
                false
            ),
            mesh(),
            dimensionedScalar(dimVolume/dimTime, Zero)
        )
    );

    tmeshPhi.ref().setOriented();

    return tmeshPhi;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// ************************************************************************* //
