/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2017 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::backwardDdtScheme

Group
    grpFvDdtSchemes

Description
    Second-order backward-differencing ddt using the current and
    two previous time-step values.

SourceFiles
    backwardDdtScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpubackwardDdtScheme_H
#define gpubackwardDdtScheme_H

#include "gpuddtScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                       Class backwardDdtScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpubackwardDdtScheme
:
    public fv::gpuddtScheme<Type>
{
    // Private Member Functions

        //- Return the current time-step
        scalar deltaT_() const;

        //- Return the previous time-step
        scalar deltaT0_() const;

        //- Return the previous time-step or GREAT if the old timestep field
        //  wasn't available in which case Euler ddt is used
        template<class GeoField>
        scalar deltaT0_(const GeoField&) const;

        //- No copy construct
        gpubackwardDdtScheme(const gpubackwardDdtScheme&) = delete;

        //- No copy assignment
        void operator=(const gpubackwardDdtScheme&) = delete;


public:

    //- Runtime type information
    TypeName("backward");


    // Constructors

        //- Construct from mesh
        gpubackwardDdtScheme(const gpufvMesh& mesh)
        :
            gpuddtScheme<Type>(mesh)
        {}

        //- Construct from mesh and Istream
        gpubackwardDdtScheme(const gpufvMesh& mesh, Istream& is)
        :
            gpuddtScheme<Type>(mesh, is)
        {
            if (is.good() && !is.eof())
            {
                is >> this->ddtPhiCoeff_;
            }

            // Ensure the old-old-time cell volumes are available
            // for moving meshes
            if (mesh.hostmesh().moving())
            {
                mesh.V00();
            }
        }


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return fv::gpuddtScheme<Type>::mesh();
        }

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensioned<Type>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        typedef typename gpuddtScheme<Type>::fluxFieldType fluxFieldType;

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<surfaceScalargpuField> meshPhi
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );
};


template<>
tmp<surfaceScalargpuField> gpubackwardDdtScheme<scalar>::fvcDdtUfCorr
(
    const GeometricgpuField<scalar, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<scalar, fvsPatchgpuField, gpusurfaceMesh>& Uf
);

template<>
tmp<surfaceScalargpuField> gpubackwardDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);

template<>
tmp<surfaceScalargpuField> gpubackwardDdtScheme<scalar>::fvcDdtUfCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& Uf
);

template<>
tmp<surfaceScalargpuField> gpubackwardDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpubackwardDdtScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
