/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::d2dt2Scheme

Description
    Abstract base class for finite volume d2dt2 schemes.

SourceFiles
    d2dt2Scheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpud2dt2Scheme_H
#define gpud2dt2Scheme_H

#include "tmp.H"
#include "dimensionedType.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

template<class Type>
class gpufvMatrix;

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                           Class d2dt2Scheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpud2dt2Scheme
:
    public refCount
{

protected:

    // Protected data

        const gpufvMesh& mesh_;


    // Private Member Functions

        //- No copy construct
        gpud2dt2Scheme(const gpud2dt2Scheme&) = delete;

        //- No copy assignment
        void operator=(const gpud2dt2Scheme&) = delete;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpud2dt2Scheme,
            Istream,
            (const gpufvMesh& mesh, Istream& schemeData),
            (mesh, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpud2dt2Scheme(const gpufvMesh& mesh)
        :
            mesh_(mesh)
        {}

        //- Construct from mesh and Istream
        gpud2dt2Scheme(const gpufvMesh& mesh, Istream&)
        :
            mesh_(mesh)
        {}


    // Selectors

        //- Return a pointer to a new d2dt2Scheme created on freestore
        static tmp<gpud2dt2Scheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpud2dt2Scheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcD2dt2
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcD2dt2
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmD2dt2
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmD2dt2
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;

        virtual tmp<gpufvMatrix<Type>> fvmD2dt2
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeFvgpuD2dt2TypeScheme(SS, Type)                                        \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpud2dt2Scheme<Type>::addIstreamConstructorToTable<SS<Type>>          \
                add##SS##Type##IstreamConstructorToTable_;                     \
        }                                                                      \
    }

#define makeFvgpuD2dt2Scheme(SS)                                                  \
                                                                               \
makeFvgpuD2dt2TypeScheme(SS, scalar)                                              \
makeFvgpuD2dt2TypeScheme(SS, vector)                                              \
makeFvgpuD2dt2TypeScheme(SS, sphericalTensor)                                     \
makeFvgpuD2dt2TypeScheme(SS, symmTensor)                                          \
makeFvgpuD2dt2TypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpud2dt2Scheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
