/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
    Copyright (C) 2020 PCOpt/NTUA
    Copyright (C) 2020 FOSS GP
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::optionList

Description
    List of finite volume options

SourceFile
    optionList.C

\*---------------------------------------------------------------------------*/

#ifndef gpufvOptionList_H
#define gpufvOptionList_H

#include "gpufvOption.H"
#include "PtrList.H"
#include "GeometricgpuField.H"
#include "geometricOneField.H"
#include "fvPatchgpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declaration of friend functions and operators

namespace fv
{
    class gpuoptionList;
}

Ostream& operator<<(Ostream& os, const fv::gpuoptionList& options);

namespace fv
{

/*---------------------------------------------------------------------------*\
                         Class optionList Declaration
\*---------------------------------------------------------------------------*/

class gpuoptionList
:
    public PtrList<fv::gpuoption>
{
protected:

    // Protected data

        //- Reference to the mesh database
        const gpufvMesh& mesh_;

        //- Time index to check that all defined sources have been applied
        label checkTimeIndex_;


    // Protected Member Functions

        //- Return "options" sub-dictionary (if present) or return dict
        static const dictionary& optionsDict(const dictionary& dict);

        //- Read options dictionary
        bool readOptions(const dictionary& dict);

        //- Check that all sources have been applied
        void checkApplied() const;

        //- Return source for equation with specified name and dimensions
        template<class Type>
        tmp<gpufvMatrix<Type>> source
        (
            GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field,
            const word& fieldName,
            const dimensionSet& ds
        );

        //- No copy construct
        gpuoptionList(const gpuoptionList&) = delete;

        //- No copy assignment
        void operator=(const gpuoptionList&) = delete;


public:

    //- Runtime type information
    TypeName("gpuoptionList");


    // Constructors

        //- Default construct from mesh
        explicit gpuoptionList(const gpufvMesh& mesh);

        //- Construct from mesh and dictionary
        gpuoptionList(const gpufvMesh& mesh, const dictionary& dict);


    //- Destructor
    virtual ~gpuoptionList() = default;


    // Member Functions

        //- Reset the source list
        void reset(const dictionary& dict);

        //- Return whether there is something to apply to the field
        bool appliesToField(const word& fieldName) const;


        // Sources

            //- Return source for equation
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation with specified name
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field,
                const word& fieldName
            );

            //- Return source for equation
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const volScalargpuField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation with specified name
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const volScalargpuField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field,
                const word& fieldName
            );

            //- Return source for equation
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const volScalargpuField& alpha,
                const volScalargpuField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation with specified name
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const volScalargpuField& alpha,
                const volScalargpuField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field,
                const word& fieldName
            );

            //- Return source for equation
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const volScalargpuField& alpha,
                const geometricOneField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const geometricOneField& alpha,
                const volScalargpuField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation
            template<class Type>
            tmp<gpufvMatrix<Type>> operator()
            (
                const geometricOneField& alpha,
                const geometricOneField& rho,
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation with second time derivative
            template<class Type>
            tmp<gpufvMatrix<Type>> d2dt2
            (
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field
            );

            //- Return source for equation with second time derivative
            template<class Type>
            tmp<gpufvMatrix<Type>> d2dt2
            (
                GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field,
                const word& fieldName
            );


        // Constraints

            //- Apply constraints to equation
            template<class Type>
            void constrain(gpufvMatrix<Type>& eqn);


        // Correction

            //- Apply correction to field
            template<class Type>
            void correct(GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& field);


        //- Post process sensitivity field related to the fvOption
        template<class Type>
        void postProcessSens
        (
            gpuField<Type>& sensField,
            const word& fieldName = word::null,
            const word& designVariablesName = word::null
        );


        // IO

            //- Read dictionary
            virtual bool read(const dictionary& dict);

            //- Write data to Ostream
            virtual bool writeData(Ostream& os) const;

            //- Ostream operator
            friend Ostream& operator<<
            (
                Ostream& os,
                const gpuoptionList& options
            );
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpufvOptionListTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
