/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
    Copyright (C) 2020 PCOpt/NTUA
    Copyright (C) 2020 FOSS GP
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvOption.H"
#include "volgpuFields.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    namespace fv
    {
        defineTypeNameAndDebug(gpuoption, 0);
        defineRunTimeSelectionTable(gpuoption, dictionary);
    }
}


// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //

void Foam::fv::gpuoption::resetApplied()
{
    applied_.resize(fieldNames_.size());
    applied_ = false;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::fv::gpuoption::gpuoption
(
    const word& name,
    const word& modelType,
    const dictionary& dict,
    const gpufvMesh& mesh
)
:
    name_(name),
    modelType_(modelType),
    mesh_(mesh),
    dict_(dict),
    coeffs_(dict.optionalSubDict(modelType + "Coeffs")),
    fieldNames_(),
    applied_(),
    active_(dict_.getOrDefault("active", true)),
    log(true)
{
    Log << incrIndent << indent << "Source: " << name_ << endl << decrIndent;
}


// * * * * * * * * * * * * * * * * * Selectors * * * * * * * * * * * * * * * //

Foam::autoPtr<Foam::fv::gpuoption> Foam::fv::gpuoption::New
(
    const word& name,
    const dictionary& coeffs,
    const gpufvMesh& mesh
)
{
    const word modelType(coeffs.get<word>("type"));

    Info<< indent
        << "Selecting finite volume options type " << modelType << endl;

    mesh.time().libs().open
    (
        coeffs,
        "libs",
        dictionaryConstructorTablePtr_
    );

    auto* ctorPtr = dictionaryConstructorTable(modelType);

    if (!ctorPtr)
    {
        FatalIOErrorInLookup
        (
            coeffs,
            "gpufvOption",
            modelType,
            *dictionaryConstructorTablePtr_
        ) << exit(FatalIOError);
    }

    return autoPtr<fv::gpuoption>(ctorPtr(name, modelType, coeffs, mesh));
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

bool Foam::fv::gpuoption::isActive()
{
    return active_;
}


Foam::label Foam::fv::gpuoption::applyToField(const word& fieldName) const
{
    return fieldNames_.find(fieldName);
}


void Foam::fv::gpuoption::checkApplied() const
{
    forAll(applied_, i)
    {
        if (!applied_[i])
        {
            WarningInFunction
                << "Source " << name_ << " defined for field "
                << fieldNames_[i] << " but never used" << endl;
        }
    }
}


void Foam::fv::gpuoption::addSup
(
    gpufvMatrix<scalar>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    gpufvMatrix<vector>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    gpufvMatrix<sphericalTensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    gpufvMatrix<symmTensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    gpufvMatrix<tensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& rho,
    gpufvMatrix<scalar>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& rho,
    gpufvMatrix<vector>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& rho,
    gpufvMatrix<sphericalTensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& rho,
    gpufvMatrix<symmTensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& rho,
    gpufvMatrix<tensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    gpufvMatrix<scalar>& eqn,
    const label fieldi
)
{
    addSup(alpha*rho, eqn, fieldi);
}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    gpufvMatrix<vector>& eqn,
    const label fieldi
)
{
    addSup(alpha*rho, eqn, fieldi);
}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    gpufvMatrix<sphericalTensor>& eqn,
    const label fieldi
)
{
    addSup(alpha*rho, eqn, fieldi);
}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    gpufvMatrix<symmTensor>& eqn,
    const label fieldi
)
{
    addSup(alpha*rho, eqn, fieldi);
}


void Foam::fv::gpuoption::addSup
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    gpufvMatrix<tensor>& eqn,
    const label fieldi
)
{
    addSup(alpha*rho, eqn, fieldi);
}


void Foam::fv::gpuoption::constrain(gpufvMatrix<scalar>& eqn, const label fieldi)
{}


void Foam::fv::gpuoption::constrain(gpufvMatrix<vector>& eqn, const label fieldi)
{}


void Foam::fv::gpuoption::constrain
(
    gpufvMatrix<sphericalTensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::constrain
(
    gpufvMatrix<symmTensor>& eqn,
    const label fieldi
)
{}


void Foam::fv::gpuoption::constrain(gpufvMatrix<tensor>& eqn, const label fieldi)
{}


void Foam::fv::gpuoption::correct(volScalargpuField& field)
{}


void Foam::fv::gpuoption::correct(volVectorgpuField& field)
{}


void Foam::fv::gpuoption::correct(volSphericalTensorgpuField& field)
{}


void Foam::fv::gpuoption::correct(volSymmTensorgpuField& field)
{}


void Foam::fv::gpuoption::correct(volTensorgpuField& field)
{}


void Foam::fv::gpuoption::postProcessSens
(
    scalargpuField& sensField,
    const word& fieldName,
    const word& designVariablesName
)
{}


void Foam::fv::gpuoption::postProcessSens
(
    vectorgpuField& sensField,
    const word& fieldName,
    const word& designVariablesName
)
{}


void Foam::fv::gpuoption::postProcessSens
(
    tensorgpuField& sensField,
    const word& fieldName,
    const word& designVariablesName
)
{}


// ************************************************************************* //
