/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2012-2018 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::MRFZoneList

Description
    List container for MRF zomes

SourceFiles
    MRFZoneList.C

\*---------------------------------------------------------------------------*/

#ifndef gpuMRFZoneList_H
#define gpuMRFZoneList_H

#include "gpufvMesh.H"
#include "dictionary.H"
#include "gpufvMatricesFwd.H"
#include "gpuMRFZone.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declaration of friend functions and operators
class gpuMRFZoneList;
Ostream& operator<<(Ostream& os, const gpuMRFZoneList& models);

/*---------------------------------------------------------------------------*\
                         Class MRFZoneList Declaration
\*---------------------------------------------------------------------------*/

class gpuMRFZoneList
:
    public PtrList<gpuMRFZone>
{
    // Private Member Functions

        //- No copy construct
        gpuMRFZoneList(const gpuMRFZoneList&) = delete;

        //- No copy assignment
        void operator=(const gpuMRFZoneList&) = delete;


protected:

    // Protected data

        //- Reference to the mesh database
        const gpufvMesh& mesh_;


public:

    //- Constructor
    gpuMRFZoneList(const gpufvMesh& mesh, const dictionary& dict);

    //- Destructor
    ~gpuMRFZoneList() = default;


    // Member Functions

        //- Return active status
        bool active(const bool warn = false) const;

        //- Reset the source list
        void reset(const dictionary& dict);

        //- Return the MRF with a given name
        const gpuMRFZone& getFromName(const word& name) const;

        //- Add the frame acceleration
        void addAcceleration
        (
            const volVectorgpuField& U,
            volVectorgpuField& ddtU
        ) const;

        //- Add the frame acceleration contribution to the momentum equation
        void addAcceleration(gpufvVectorMatrix& UEqn) const;

        //- Add the frame acceleration contribution to the momentum equation
        void addAcceleration
        (
            const volScalargpuField& rho,
            gpufvVectorMatrix& UEqn
        ) const;

        //- Return the frame acceleration
        tmp<volVectorgpuField> DDt
        (
            const volVectorgpuField& U
        ) const;

        //- Return the frame acceleration
        tmp<volVectorgpuField> DDt
        (
            const volScalargpuField& rho,
            const volVectorgpuField& U
        ) const;

        //- Make the given absolute velocity relative within the MRF region
        void makeRelative(volVectorgpuField& U) const;

        //- Make the given absolute flux relative within the MRF region
        void makeRelative(surfaceScalargpuField& phi) const;

        //- Return the given absolute flux relative within the MRF region
        tmp<surfaceScalargpuField> relative
        (
            const tmp<surfaceScalargpuField>& phi
        ) const;

        //- Return the given absolute boundary flux relative within
        //  the MRF region
        tmp<FieldField<fvsPatchgpuField, scalar>> relative
        (
            const tmp<FieldField<fvsPatchgpuField, scalar>>& tphi
        ) const;

        //- Return the given absolute patch flux relative within
        //  the MRF region
        tmp<gpuField<scalar>> relative
        (
            const tmp<gpuField<scalar>>& tphi,
            const label patchi
        ) const;

        //- Make the given absolute mass-flux relative within the MRF region
        void makeRelative
        (
            const surfaceScalargpuField& rho,
            surfaceScalargpuField& phi
        ) const;

        //- Make the given relative velocity absolute within the MRF region
        void makeAbsolute(volVectorgpuField& U) const;

        //- Make the given relative flux absolute within the MRF region
        void makeAbsolute(surfaceScalargpuField& phi) const;

        //- Return the given relative flux absolute within the MRF region
        tmp<surfaceScalargpuField> absolute
        (
            const tmp<surfaceScalargpuField>& phi
        ) const;

        //- Make the given relative mass-flux absolute within the MRF region
        void makeAbsolute
        (
            const surfaceScalargpuField& rho,
            surfaceScalargpuField& phi
        ) const;

        //- Correct the boundary velocity for the rotation of the MRF region
        void correctBoundaryVelocity(volVectorgpuField& U) const;

        //- Correct the boundary flux for the rotation of the MRF region
        void correctBoundaryFlux
        (
            const volVectorgpuField& U,
            surfaceScalargpuField& phi
        ) const;

        //- Filter-out the MRF region contribution from the given field
        // setting the corresponding values to zero
        template<class Type>
        tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> zeroFilter
        (
            const tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>& tphi
        ) const;

        //- Update MRFZone faces if the mesh topology changes
        void update();


        // I-O

            //- Read dictionary
            bool read(const dictionary& dict);

            //- Write data to Ostream
            bool writeData(Ostream& os) const;

            //- Ostream operator
            friend Ostream& operator<<
            (
                Ostream& os,
                const gpuMRFZoneList& models
            );
};

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuMRFZoneListTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
