/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "GPBiCG.H"
#include "PrecisionAdaptor.H"
#include "lduMatrixSolverFunctors.H"
#include "PCGCache.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(GPBiCG, 0);

    gpulduMatrix::solver::addasymMatrixConstructorToTable<GPBiCG>
        addGPBiCGAsymMatrixConstructorToTable_;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::GPBiCG::GPBiCG
(
    const word& fieldName,
    const gpulduMatrix& matrix,
    const FieldField<gpuField, scalar>& interfaceBouCoeffs,
    const FieldField<gpuField, scalar>& interfaceIntCoeffs,
    const lduInterfacegpuFieldPtrsList& interfaces,
    const dictionary& solverControls
)
:
    gpulduMatrix::solver
    (
        fieldName,
        matrix,
        interfaceBouCoeffs,
        interfaceIntCoeffs,
        interfaces,
        solverControls
    )
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::solverPerformance Foam::GPBiCG::solve
(
    scalargpuField& psi_s,
    const scalargpuField& source,
    const direction cmpt
) const
{
    // --- Setup class containing solver performance data
    solverPerformance solverPerf
    (
        gpulduMatrix::preconditioner::getName(controlDict_) + typeName,
        fieldName_
    );

    const label nCells = psi_s.size();

    scalargpuField pA(PCGCache::pA(matrix_.level(),nCells),nCells);

    scalargpuField wA(PCGCache::wA(matrix_.level(),nCells),nCells);

    // --- Calculate A.psi
    matrix_.Amul(wA, psi_s, interfaceBouCoeffs_, interfaces_, cmpt);

    // --- Calculate initial residual field
    scalargpuField rA(PCGCache::rA(matrix_.level(),nCells),nCells);
    thrust::transform
    (
        source.begin(),
        source.end(),
        wA.begin(),
        rA.begin(),
        minusOp<scalar>()
    );

    /*matrix().setResidualField
    (
        ConstPrecisionAdaptor<scalar, solveScalar>(rA)(),
        fieldName_,
        true
    );*/

    // --- Calculate normalisation factor
    const solveScalar normFactor = this->normFactor(psi_s, source, wA, pA);

    if ((log_ >= 2) || (gpulduMatrix::debug >= 2))
    {
        Info<< "   Normalisation factor = " << normFactor << endl;
    }

    // --- Calculate normalised residual norm
    solverPerf.initialResidual() =
        gSumMag(rA, matrix().mesh().comm())
       /normFactor;
    solverPerf.finalResidual() = solverPerf.initialResidual();

    // --- Check convergence, solve if not converged
    if
    (
        minIter_ > 0
     || !solverPerf.checkConvergence(tolerance_, relTol_, log_)
    )
    {
		scalargpuField pT(PCGCache::pT(matrix_.level(),nCells),nCells);
    	//pT = 0.0;
    	scalargpuField wT(PCGCache::wT(matrix_.level(),nCells),nCells);
		
        // --- Calculate T.psi
        matrix_.Tmul(wT, psi_s, interfaceIntCoeffs_, interfaces_, cmpt);

        // --- Calculate initial transpose residual field
        scalargpuField rT(PCGCache::rT(matrix_.level(),nCells),nCells);
		thrust::transform
    	(
        	source.begin(),
        	source.end(),
       	 	wT.begin(),
        	rT.begin(),
       	 	minusOp<scalar>()
    	);

        // --- Initial value not used
        solveScalar wArT = 0;
		
        //solveScalar wArT = solverPerf.great_;
    	//solveScalar wArTold = wArT;

        // --- Select and construct the preconditioner
        autoPtr<gpulduMatrix::preconditioner> preconPtr =
        gpulduMatrix::preconditioner::New
        (
            *this,
            controlDict_
        );

        // --- Solver iteration
        do
        {
            // --- Store previous wArT
            const solveScalar wArTold = wArT;

            // --- Precondition residuals
            preconPtr->precondition(wA, rA, cmpt);
            preconPtr->preconditionT(wT, rT, cmpt);

            // --- Update search directions:
            wArT = gSumProd(wA, rT, matrix().mesh().comm());

            if (solverPerf.nIterations() == 0)
            {
                thrust::copy(wA.begin(),wA.end(),pA.begin());
                thrust::copy(wT.begin(),wT.end(),pT.begin());
            }
            else
            {
                const solveScalar beta = wArT/wArTold;
				
                thrust::transform
                (
                    wA.begin(),
                    wA.end(),
                    pA.begin(),
                    pA.begin(),
                    wAPlusBetaPAFunctor(beta)
                );

                thrust::transform
                (
                    wT.begin(),
                    wT.end(),
                    pT.begin(),
                    pT.begin(),
                    wAPlusBetaPAFunctor(beta)
                );
            }


            // --- Update preconditioned residuals
            matrix_.Amul(wA, pA, interfaceBouCoeffs_, interfaces_, cmpt);
            matrix_.Tmul(wT, pT, interfaceIntCoeffs_, interfaces_, cmpt);

            const solveScalar wApT = gSumProd(wA, pT, matrix().mesh().comm());

            // --- Test for singularity
            if (solverPerf.checkSingularity(mag(wApT)/normFactor))
            {
                break;
            }

            // --- Update solution and residual:

            const solveScalar alpha = wArT/wApT;

            thrust::transform
            (
                psi_s.begin(),
                psi_s.end(),
                pA.begin(),
                psi_s.begin(),
                psiPlusAlphaPAFunctor(alpha)
            );

            thrust::transform
            (
                rA.begin(),
                rA.end(),
                wA.begin(),
                rA.begin(),
                rAMinusAlphaWAFunctor(alpha)
            );

            thrust::transform
            (
                rT.begin(),
                rT.end(),
                wT.begin(),
                rT.begin(),
                rAMinusAlphaWAFunctor(alpha)
            );			

            solverPerf.finalResidual() =
                gSumMag(rA, matrix().mesh().comm())
               /normFactor;
        } while
        (
            (
              ++solverPerf.nIterations() < maxIter_
            && !solverPerf.checkConvergence(tolerance_, relTol_, log_)
            )
         || solverPerf.nIterations() < minIter_
        );
    }

    // Recommend PBiCGStab if PBiCG fails to converge
    if (solverPerf.nIterations() > max(defaultMaxIter_, maxIter_))
    {
        FatalErrorInFunction
            << "PBiCG has failed to converge within the maximum number"
               " of iterations " << max(defaultMaxIter_, maxIter_) << nl
            << "    Please try the more robust PBiCGStab solver."
            << exit(FatalError);
    }

    /*matrix().setResidualField
    (
        ConstPrecisionAdaptor<scalar, solveScalar>(rA)(),
        fieldName_,
        false
    );*/
	
    return solverPerf;
}


// ************************************************************************* //
