/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Description
    lduMatrix member operations.

\*---------------------------------------------------------------------------*/

#include "gpulduMatrix.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

void Foam::gpulduMatrix::sumDiag()
{
    const scalargpuField& Lower = const_cast<const gpulduMatrix&>(*this).gpuLower();
    const scalargpuField& Upper = const_cast<const gpulduMatrix&>(*this).gpuUpper();
    scalargpuField& Diag = gpuDiag();

    matrixOperation
    (
        Diag.begin(),
        Diag,
        lduAddr(),
        matrixCoeffsFunctor<scalar,unityOp<scalar> >
        (
            Lower.data(),
            unityOp<scalar>()
        ),
        matrixCoeffsFunctor<scalar,unityOp<scalar> >
        (
            Upper.data(),
            unityOp<scalar>()
        )
    ); 
}


void Foam::gpulduMatrix::negSumDiag()
{
    const scalargpuField& Lower = const_cast<const gpulduMatrix&>(*this).gpuLower();
    const scalargpuField& Upper = const_cast<const gpulduMatrix&>(*this).gpuUpper();
    scalargpuField& Diag = gpuDiag();

    matrixOperation
    (
        Diag.begin(),
        Diag,
        lduAddr(),
        matrixCoeffsFunctor<scalar,negateUnaryOperatorFunctor<scalar,scalar> >
        (
            Lower.data(),
            negateUnaryOperatorFunctor<scalar,scalar>()
        ),
        matrixCoeffsFunctor<scalar,negateUnaryOperatorFunctor<scalar,scalar> >
        (
            Upper.data(),
            negateUnaryOperatorFunctor<scalar,scalar>()
        )
    ); 
}

void Foam::gpulduMatrix::sumMagOffDiag
(
    scalargpuField& sumOff
) const
{
    const scalargpuField& Lower = const_cast<const gpulduMatrix&>(*this).gpuLower();
    const scalargpuField& Upper = const_cast<const gpulduMatrix&>(*this).gpuUpper();

    matrixOperation
    (
        sumOff.begin(),
        sumOff,
        lduAddr(),
        matrixCoeffsFunctor<scalar,magUnaryFunctionFunctor<scalar,scalar> >
        (
            Upper.data(),
            magUnaryFunctionFunctor<scalar,scalar>()
        ),
        matrixCoeffsFunctor<scalar,magUnaryFunctionFunctor<scalar,scalar> >
        (
            Lower.data(),
            magUnaryFunctionFunctor<scalar,scalar>()
        )
    ); 
}

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

void Foam::gpulduMatrix::operator=(const gpulduMatrix& A)
{
    if (this == &A)
    {
        return;  // Self-assignment is a no-op
    }

    if (A.lowerPtr_)
    {
        gpuLower() = A.gpuLower();
    }
    else if (lowerPtr_)
    {
        delete lowerPtr_;
        lowerPtr_ = nullptr;
    }

    if (A.upperPtr_)
    {
        gpuUpper() = A.gpuUpper();
    }
    else if (upperPtr_)
    {
        delete upperPtr_;
        upperPtr_ = nullptr;
    }

    if (A.diagPtr_)
    {
        gpuDiag() = A.gpuDiag();
    }
}


void Foam::gpulduMatrix::negate()
{
    if (lowerPtr_)
    {
        lowerPtr_->negate();
    }

    if (upperPtr_)
    {
        upperPtr_->negate();
    }

    if (diagPtr_)
    {
        diagPtr_->negate();
    }
}


void Foam::gpulduMatrix::operator+=(const gpulduMatrix& A)
{
    if (A.diagPtr_)
    {
        gpuDiag() += A.gpuDiag();
    }

    if (symmetric() && A.symmetric())
    {
        gpuUpper() += A.gpuUpper();
    }
    else if (symmetric() && A.asymmetric())
    {
        if (upperPtr_)
        {
            gpuLower();
        }
        else
        {
            gpuUpper();
        }

        gpuUpper() += A.gpuUpper();
        gpuLower() += A.gpuLower();
    }
    else if (asymmetric() && A.symmetric())
    {
        if (A.upperPtr_)
        {
            gpuLower() += A.gpuUpper();
            gpuUpper() += A.gpuUpper();
        }
        else
        {
            gpuLower() += A.gpuLower();
            gpuUpper() += A.gpuLower();
        }

    }
    else if (asymmetric() && A.asymmetric())
    {
        gpuLower() += A.gpuLower();
        gpuUpper() += A.gpuUpper();
    }
    else if (diagonal())
    {
        if (A.upperPtr_)
        {
            gpuUpper() = A.gpuUpper();
        }

        if (A.lowerPtr_)
        {
            gpuLower() = A.gpuLower();
        }
    }
    else if (A.diagonal())
    {
    }
    else
    {
        if (debug > 1)
        {
            WarningInFunction
                << "Unknown matrix type combination" << nl
                << "    this :"
                << " diagonal:" << diagonal()
                << " symmetric:" << symmetric()
                << " asymmetric:" << asymmetric() << nl
                << "    A    :"
                << " diagonal:" << A.diagonal()
                << " symmetric:" << A.symmetric()
                << " asymmetric:" << A.asymmetric()
                << endl;
        }
    }
}


void Foam::gpulduMatrix::operator-=(const gpulduMatrix& A)
{
    if (A.diagPtr_)
    {
        gpuDiag() -= A.gpuDiag();
    }

    if (symmetric() && A.symmetric())
    {
        gpuUpper() -= A.gpuUpper();
    }
    else if (symmetric() && A.asymmetric())
    {
        if (upperPtr_)
        {
            gpuLower();
        }
        else
        {
            gpuUpper();
        }

        gpuUpper() -= A.gpuUpper();
        gpuLower() -= A.gpuLower();
    }
    else if (asymmetric() && A.symmetric())
    {
        if (A.upperPtr_)
        {
            gpuLower() -= A.gpuUpper();
            gpuUpper() -= A.gpuUpper();
        }
        else
        {
            gpuLower() -= A.gpuLower();
            gpuUpper() -= A.gpuLower();
        }

    }
    else if (asymmetric() && A.asymmetric())
    {
        gpuLower() -= A.gpuLower();
        gpuUpper() -= A.gpuUpper();
    }
    else if (diagonal())
    {
        if (A.upperPtr_)
        {
            gpuUpper() = -A.gpuUpper();
        }

        if (A.lowerPtr_)
        {
            gpuLower() = -A.gpuLower();
        }
    }
    else if (A.diagonal())
    {
    }
    else
    {
        if (debug > 1)
        {
            WarningInFunction
                << "Unknown matrix type combination" << nl
                << "    this :"
                << " diagonal:" << diagonal()
                << " symmetric:" << symmetric()
                << " asymmetric:" << asymmetric() << nl
                << "    A    :"
                << " diagonal:" << A.diagonal()
                << " symmetric:" << A.symmetric()
                << " asymmetric:" << A.asymmetric()
                << endl;
        }
    }
}


void Foam::gpulduMatrix::operator*=(const scalargpuField& sf)
{
    if (diagPtr_)
    {
        *diagPtr_ *= sf;
    }

    // Non-uniform scaling causes a symmetric matrix
    // to become asymmetric
    if (symmetric() || asymmetric())
    {
        scalargpuField& upper = this->gpuUpper();
        scalargpuField& lower = this->gpuLower();

        const labelgpuList& l = lduAddr().lowerAddr();
        const labelgpuList& u = lduAddr().upperAddr();

        thrust::transform
        (
            upper.begin(),
            upper.end(),
            thrust::make_permutation_iterator(sf.begin(),l.begin()),
            upper.begin(),
            multiplyOperatorFunctor<scalar,scalar,scalar>()
        );

        thrust::transform
        (
            lower.begin(),
            lower.end(),
            thrust::make_permutation_iterator(sf.begin(),u.begin()),
            lower.begin(),
            multiplyOperatorFunctor<scalar,scalar,scalar>()
        );
    }
}

void Foam::gpulduMatrix::operator*=(scalar s)
{
    if (diagPtr_)
    {
        *diagPtr_ *= s;
    }

    if (upperPtr_)
    {
        *upperPtr_ *= s;
    }

    if (lowerPtr_)
    {
        *lowerPtr_ *= s;
    }
}


// ************************************************************************* //
