/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2013 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::lduInterfaceField

Description
    An abstract base class for implicitly-coupled interface fields
    e.g. processor and cyclic patch fields.

SourceFiles
    lduInterfaceField.C

\*---------------------------------------------------------------------------*/

#ifndef lduInterfacegpuField_H
#define lduInterfacegpuField_H

#include "gpulduInterface.H"
#include "primitiveFieldsFwd.H"
#include "Pstream.H"
#include "gpulduAddressing.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declarations
class gpulduMatrix;

/*---------------------------------------------------------------------------*\
                     Class lduInterfaceField Declaration
\*---------------------------------------------------------------------------*/

class lduInterfacegpuField
{
    // Private Data

        //- Reference to the coupled patch this field is defined for
        const gpulduInterface& interface_;

        //- Update index used so that updateInterfaceMatrix is called only once
        //  during the construction of the matrix
        bool updatedMatrix_;


    // Private Member Functions

        //- No copy construct
        lduInterfacegpuField(const lduInterfacegpuField&) = delete;

        //- No copy assignment
        void operator=(const lduInterfacegpuField&) = delete;


public:

    //- Runtime type information
    TypeName("lduInterfacegpuField");


    // Constructors

        //- Construct given coupled patch
        explicit lduInterfacegpuField(const gpulduInterface& patch)
        :
            interface_(patch),
            updatedMatrix_(false)
        {}


    //- Destructor
    virtual ~lduInterfacegpuField() = default;


    // Member Functions

        // Access

            //- Return the interface
            const gpulduInterface& interface() const
            {
                return interface_;
            }

            //- Return the interface type
            virtual const word& interfaceFieldType() const
            {
                return type();
            }


        // Coupled interface matrix update

            //- Whether matrix has been updated
            bool updatedMatrix() const
            {
                return updatedMatrix_;
            }

            //- Whether matrix has been updated
            bool& updatedMatrix()
            {
                return updatedMatrix_;
            }

            //- Is all data available
            virtual bool ready() const
            {
                return true;
            }

            //- Initialise neighbour matrix update.
            //- Add/subtract coupled contributions to matrix
            virtual void initInterfaceMatrixUpdate
            (
                scalargpuField& result,
                const bool add,
                const gpulduAddressing&,
                const label interfacei,
                const scalargpuField& psiInternal,
                const scalargpuField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const
            {}

            //- Update result field based on interface functionality.
            //- Add/subtract coupled contributions to matrix
           virtual void updateInterfaceMatrix
            (
                scalargpuField& result,
                const bool add,
                const gpulduAddressing&,
                const label interfacei,
                const scalargpuField& psiInternal,
                const scalargpuField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const = 0;

            //- Add/subtract weighted contributions to internal field
            template<class Type>
            void addToInternalField
            (
                gpuField<Type>& result,
                const bool add,
                const label& patchi,
                const gpulduAddressing& lduAddr,
                const scalargpuField& coeffs,
                const gpuField<Type>& vals
            ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "lduInterfacegpuFieldTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
