/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2016-2017 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "Maxwell.H"
#include "gpufvOptions.H"
#include "uniformDimensionedFields.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace laminarModels
{

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class BasicTurbulenceModel>
Maxwell<BasicTurbulenceModel>::Maxwell
(
    const alphaField& alpha,
    const rhoField& rho,
    const volVectorgpuField& U,
    const surfaceScalargpuField& alphaRhoPhi,
    const surfaceScalargpuField& phi,
    const transportModel& transport,
    const word& propertiesName,
    const word& type
)
:
    laminarModel<BasicTurbulenceModel>
    (
        type,
        alpha,
        rho,
        U,
        alphaRhoPhi,
        phi,
        transport,
        propertiesName
    ),

    nuM_("nuM", dimViscosity, this->coeffDict_),

    lambda_("lambda", dimTime, this->coeffDict_),

    sigma_
    (
        IOobject
        (
            IOobject::groupName("sigma", alphaRhoPhi.group()),
            this->runTime_.timeName(),
            this->mesh_.hostmesh(),
            IOobject::MUST_READ,
            IOobject::AUTO_WRITE
        ),
        this->mesh_
    )
{
    if (type == typeName)
    {
        this->printCoeffs(type);
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class BasicTurbulenceModel>
bool Maxwell<BasicTurbulenceModel>::read()
{
    if (laminarModel<BasicTurbulenceModel>::read())
    {
        nuM_.readIfPresent(this->coeffDict());
        lambda_.readIfPresent(this->coeffDict());

        return true;
    }

    return false;
}


template<class BasicTurbulenceModel>
tmp<Foam::volSymmTensorgpuField>
Maxwell<BasicTurbulenceModel>::R() const
{
    return sigma_;
}


template<class BasicTurbulenceModel>
tmp<Foam::volSymmTensorgpuField>
Maxwell<BasicTurbulenceModel>::devRhoReff() const
{
    return devRhoReff(this->U_);
}


template<class BasicTurbulenceModel>
tmp<Foam::volSymmTensorgpuField>
Maxwell<BasicTurbulenceModel>::devRhoReff(const volVectorgpuField& U) const
{
    return tmp<volSymmTensorgpuField>
    (
        new volSymmTensorgpuField
        (
            IOobject
            (
                IOobject::groupName("devRhoReff", this->alphaRhoPhi_.group()),
                this->runTime_.timeName(),
                this->mesh_.hostmesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE
            ),
            this->alpha_*this->rho_*sigma_
          - (this->alpha_*this->rho_*this->nu())
           *dev(twoSymm(fvc::grad(U)))
        )
    );
}


template<class BasicTurbulenceModel>
tmp<Foam::gpufvVectorMatrix>
Maxwell<BasicTurbulenceModel>::divDevRhoReff
(
    volVectorgpuField& U
) const
{
    return
    (
        fvc::div
        (
            this->alpha_*this->rho_*this->nuM_*fvc::grad(U)
        )
      + fvc::div(this->alpha_*this->rho_*sigma_)
      - fvc::div(this->alpha_*this->rho_*this->nu()*dev2(T(fvc::grad(U))))
      - fvm::laplacian(this->alpha_*this->rho_*nu0(), U)
    );
}


template<class BasicTurbulenceModel>
tmp<Foam::gpufvVectorMatrix>
Maxwell<BasicTurbulenceModel>::divDevRhoReff
(
    const volScalargpuField& rho,
    volVectorgpuField& U
) const
{
    return
    (
        fvc::div
        (
            this->alpha_*rho*this->nuM_*fvc::grad(U)
        )
      + fvc::div(this->alpha_*rho*sigma_)
      - fvc::div(this->alpha_*rho*this->nu()*dev2(T(fvc::grad(U))))
      - fvm::laplacian(this->alpha_*rho*nu0(), U)
    );
}


template<class BasicTurbulenceModel>
void Maxwell<BasicTurbulenceModel>::correct()
{
    // Local references
    const alphaField& alpha = this->alpha_;
    const rhoField& rho = this->rho_;
    const surfaceScalargpuField& alphaRhoPhi = this->alphaRhoPhi_;
    const volVectorgpuField& U = this->U_;
    volSymmTensorgpuField& sigma = this->sigma_;
    fv::gpuoptions& fvOptions(fv::gpuoptions::New(this->mesh_));

    laminarModel<BasicTurbulenceModel>::correct();

    tmp<volTensorgpuField> tgradU(fvc::grad(U));
    const volTensorgpuField& gradU = tgradU();

    uniformDimensionedScalarField rLambda
    (
        IOobject
        (
            IOobject::groupName("rLambda", this->alphaRhoPhi_.group()),
            this->runTime_.constant(),
            this->mesh_.hostmesh()
        ),
        1.0/(lambda_)
    );

    // Note sigma is positive on lhs of momentum eqn
    volSymmTensorgpuField P
    (
        twoSymm(sigma & gradU)
      - nuM_*rLambda*twoSymm(gradU)
    );

    // Viscoelastic stress equation
    tmp<gpufvSymmTensorMatrix> sigmaEqn
    (
        fvm::ddt(alpha, rho, sigma)
      + fvm::div(alphaRhoPhi, sigma)
      + fvm::Sp(alpha*rho*rLambda, sigma)
     ==
        alpha*rho*P
      + fvOptions(alpha, rho, sigma)
    );

    sigmaEqn.ref().relax();
    fvOptions.constrain(sigmaEqn.ref());
    solve(sigmaEqn);
    fvOptions.correct(sigma_);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace laminarModels
} // End namespace Foam

// ************************************************************************* //
