/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2013 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::distributionModels::normal

Description
    Particle-size distribution model wherein random samples are drawn
    from the doubly-truncated univariate normal probability density function:

    \f[
        f(x; \mu, \sigma, A, B) =
            \frac{1}{\sigma}
            \frac{
                \phi \left( \frac{x - \mu}{\sigma} \right)
            }{
                \Phi \left( \frac{B - \mu}{\sigma} \right)
              - \Phi \left( \frac{A - \mu}{\sigma} \right)
            }
    \f]
    where

    \vartable
      f(x; \mu, \sigma, A, B) | Doubly-truncated univariate normal distribution
      \mu         | Mean of the parent general normal distribution
      \sigma      | Standard deviation of the parent general normal distribution
      \phi(\cdot) | General normal probability density function
      \Phi(\cdot) | General normal cumulative distribution function
      x           | Sample
      A           | Minimum of the distribution
      B           | Maximum of the distribution
    \endvartable

    Constraints:
    - \f$ \infty > B > A > 0 \f$
    - \f$ x \in [B,A] \f$
    - \f$ \sigma^2 > 0 \f$

    Random samples are generated by the inverse transform sampling technique
    by using the quantile function of the doubly-truncated univariate normal
    probability density function:

    \f[
        x = \mu + \sigma \sqrt{2} \, {erf}^{-1} \left( 2 p - 1 \right)
    \f]
    with

    \f[
        p = u \,
            \left(
                \Phi\left(
                    \frac{B - \mu}{\sigma}
                    \right)
              - \Phi\left(
                    \frac{A - \mu}{\sigma}
                    \right)
            \right)
          + \Phi\left( \frac{A - \mu}{\sigma} \right)
    \f]

    \f[
        \Phi(\xi) =
            \frac{1}{2}
            \left(
                1 + {erf}\left(\frac{\xi - \mu}{\sigma \sqrt{2} }\right)
            \right)
    \f]
    where \f$ u \f$ is sample drawn from the uniform probability
    density function on the unit interval \f$ (0, 1) \f$.

    Reference:
    \verbatim
        Governing expressions (tag:B):
            Burkardt, J. (2014).
            The truncated normal distribution.
            Department of Scientific Computing Website,
            Florida State University, 1-35.
            URL:people.sc.fsu.edu/~jburkardt/presentations/truncated_normal.pdf
            (Retrieved on: 19 Feb 2021)
    \endverbatim

Usage
    Minimal example by using \c constant/\<CloudProperties\>:
    \verbatim
    subModels
    {
        injectionModels
        {
            <name>
            {
                ...

                sizeDistribution
                {
                    type        normal;
                    normalDistribution
                    {
                        mu        <mean>;
                        sigma     <stardard deviation>;
                        minValue  <min>;
                        maxValue  <max>;
                    }
                }
            }
        }
    }
    \endverbatim

    where the entries mean:
    \table
      Property   | Description                       | Type | Reqd   | Deflt
      type       | Type name: normal                 | word | yes    | -
      normalDistribution | Distribution settings     | dict | yes    | -
      mu         | Mean of the parent general normal distribution <!--
                 -->                                 | scalar | yes  | -
      sigma      | Standard deviation of the parent general normal <!--
                 --> distribution                    | scalar | yes  | -
      minValue   | Minimum of the distribution       | scalar | yes  | -
      maxValue   | Maximum of the distribution       | scalar | yes  | -
    \endtable

Note
  - The mean and standard deviation of the parent general normal probability
  distribution function (i.e. input) are not the same with those of the
  truncated probability distribution function.

SourceFiles
    normal.C

\*---------------------------------------------------------------------------*/

#ifndef distributionModels_normal_H
#define distributionModels_normal_H

#include "distributionModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace distributionModels
{

/*---------------------------------------------------------------------------*\
                           Class normal Declaration
\*---------------------------------------------------------------------------*/

class normal
:
    public distributionModel
{
    // Private Data

        //- Mean of the parent general normal distribution
        scalar mu_;

        //- Standard deviation of the parent general normal distribution
        scalar sigma_;


public:

    //- Runtime type information
    TypeName("normal");


    // Constructors

        //- Construct from components
        normal(const dictionary& dict, Random& rndGen);

        //- Copy construct
        normal(const normal& p);

        //- Construct and return a clone
        virtual autoPtr<distributionModel> clone() const
        {
            return autoPtr<distributionModel>(new normal(*this));
        }

        //- No copy assignment
        void operator=(const normal&) = delete;


    //- Destructor
    virtual ~normal() = default;


    // Member Functions

        //- Sample the distribution
        virtual scalar sample() const;

        //- Return the theoretical mean of the distribution
        virtual scalar meanValue() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace distributionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
