/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvGeometryScheme

Description
    Abstract base class for geometry calculation schemes.

SourceFiles
    fvGeometryScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpufvGeometryScheme_H
#define gpufvGeometryScheme_H

#include "tmp.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"
#include "pointField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class gpufvMesh;

/*---------------------------------------------------------------------------*\
                      Class fvGeometryScheme Declaration
\*---------------------------------------------------------------------------*/

class gpufvGeometryScheme
:
    public refCount
{
    // Private Member Functions

        //- No copy construct
        gpufvGeometryScheme(const gpufvGeometryScheme&) = delete;

        //- No copy assignment
        void operator=(const gpufvGeometryScheme&) = delete;


protected:

        //- Hold reference to mesh
        const gpufvMesh& mesh_;


public:

    //- Runtime type information
    TypeName("gpufvGeometryScheme");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpufvGeometryScheme,
            dict,
            (
                const gpufvMesh& mesh,
                const dictionary& dict
            ),
            (mesh, dict)
        );


    // Constructors

        //- Construct from mesh
        gpufvGeometryScheme(const gpufvMesh& mesh, const dictionary& dict)
        :
            mesh_(mesh)
        {}


    // Selectors

        //- Return new tmp interpolation scheme
        static tmp<gpufvGeometryScheme> New
        (
            const gpufvMesh& mesh,
            const dictionary& dict,
            const word& defaultScheme
        );


    //- Destructor
    virtual ~gpufvGeometryScheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        //- Update basic geometric properties from provided points
        virtual void movePoints()
        {}

        //- Return linear difference weighting factors
        virtual tmp<surfaceScalargpuField> weights() const = 0;

        //- Return cell-centre difference coefficients
        virtual tmp<surfaceScalargpuField> deltaCoeffs() const = 0;

        //- Return non-orthogonal cell-centre difference coefficients
        virtual tmp<surfaceScalargpuField> nonOrthDeltaCoeffs() const = 0;

        //- Return non-orthogonality correction vectors
        virtual tmp<surfaceVectorgpuField> nonOrthCorrectionVectors() const = 0;

        ////- Selector for wall distance method. WIP. Ideally return wall
        ////  distance or meshObject?
        //virtual autoPtr<patchDistMethod> newPatchDistMethod
        //(
        //    const dictionary& dict,
        //    const labelHashSet& patchIDs,
        //    const word& defaultPatchDistMethod
        //) const = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
